#pragma once

#include <api/common.h>
#include <xtable/resharding/query.h>

namespace yxiva { namespace hub { namespace api { namespace resharding {

namespace {

template <typename QueryTraits, typename Callback, int... S>
void execute(
    shared_ptr<XTable> xtable,
    task_context_ptr ctx,
    typename QueryTraits::params&& params,
    Callback&& cb,
    yplatform::util::sequence<S...>)
{
    (xtable.get()->*QueryTraits::method())(
        ctx,
        std::get<S>(std::forward<typename QueryTraits::params>(params))...,
        std::forward<Callback>(cb));
}

template <typename SendCallback, typename CallbackParams>
struct packing_callback
{
    std::decay_t<SendCallback> send_cb;

    template <typename... Args>
    void operator()(const error_code& ec, Args&&... args)
    {
        xtable::resharding::response_body<CallbackParams> params{
            ec.value(), CallbackParams{ std::forward<Args>(args)... }
        };
        send_cb(yxiva::pack(params));
    }
};

#define CASE_EXECUTE(method)                                                                       \
    case q_type::method:                                                                           \
    {                                                                                              \
        q::method##_traits::params unpacked_params;                                                \
        try                                                                                        \
        {                                                                                          \
            yxiva::unpack(packed_params, unpacked_params);                                         \
        }                                                                                          \
        catch (const std::exception& e)                                                            \
        {                                                                                          \
            WEB_RESPONSE_CODE_LOG_G(                                                               \
                error,                                                                             \
                stream,                                                                            \
                ymod_webserver::codes::bad_request,                                                \
                "failed to unpack xtable query params");                                           \
        }                                                                                          \
        execute<q::method##_traits>(                                                               \
            xtable,                                                                                \
            stream->ctx(),                                                                         \
            std::move(unpacked_params),                                                            \
            packing_callback<Callback, q::method##_traits::cb_params>{                             \
                std::forward<Callback>(cb) },                                                      \
            typename yplatform::util::gensequence<std::tuple_size<                                 \
                typename std::decay<q::method##_traits::params>::type>::value>::type());           \
        break;                                                                                     \
    }

template <typename Callback>
void execute_packed_query(
    shared_ptr<XTable> xtable,
    ymod_webserver::http::stream_ptr stream,
    xtable::resharding::query_type type,
    const std::string& packed_params,
    Callback&& cb)
{
    namespace q = xtable::resharding;
    using q_type = xtable::resharding::query_type;

    switch (type)
    {
        CASE_EXECUTE(subscribe);
        CASE_EXECUTE(subscribe_mobile);
        CASE_EXECUTE(unsubscribe);
        CASE_EXECUTE(unsubscribe_mobile);
        CASE_EXECUTE(unsubscribe_overlapped);
        CASE_EXECUTE(batch_unsubscribe);
        CASE_EXECUTE(update);
        CASE_EXECUTE(update_uidset);
        CASE_EXECUTE(update_callback);
    };
}

#undef CASE_EXECUTE
} // namespace

struct execute_xtable : public api_coroutine
{
    std::shared_ptr<state> hub;
    stream_ptr stream;
    xtable::resharding::query_type query_type;

    void operator()(const string& packed_cb_params = {})
    {
        auto req = stream->request();

        reenter(this)
        {
            yield execute_packed_query(
                hub->xtable,
                stream,
                query_type,
                string(req->raw_body.begin(), req->raw_body.end()),
                *this);
            stream->result(ymod_webserver::codes::ok, packed_cb_params);
        }
    }
};

}}}}
