#pragma once

#include "common.h"
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace yxiva { namespace hub { namespace api {

namespace detail {
void correct_range(
    local_id_t range_bottom,
    local_id_t range_top,
    local_id_t& position,
    unsigned& count);
operation::result extract_appname(const sub_t& sub, string& appname);
bool push_tokens_match(const sub_t& lhs, const sub_t& rhs);
bool devices_appnames_match(const sub_t& lhs, const sub_t& rhs);

struct deduplicate_coro
{
    std::shared_ptr<state> hub;
    task_context_ptr ctx;
    string uid;
    string service;
    std::function<void(const boost::system::error_code&, sub_list)> handler;

    void operator()(
        yplatform::yield_context<deduplicate_coro> yield_ctx,
        boost::system::error_code ec = boost::system::error_code())
    {
        reenter(yield_ctx)
        {
            yield hub->xtable->find(ctx, uid, service, {}, yield_ctx);
            if (ec)
            {
                hub->transport_log->deduplicate_list_failed(ctx, uid, service, ec.message());
                handler(ec, {});
                return;
            }
            erase_non_mobile_subs();
            for (auto& sub : subs_)
            {
                if (marked_as_duplicate(sub))
                {
                    continue;
                }
                mark_duplicates(sub, subs_);
            }
            if (duplicate_subs_.empty())
            {
                handler(ec, {});
                return;
            }
            yield hub->xtable->batch_unsubscribe(
                ctx, uid, service, get_ids(duplicate_subs_), time(nullptr), yield_ctx);
            if (ec)
            {
                hub->transport_log->deduplicate_unsubscribe_failed(ctx, uid, service, ec.message());
                handler(ec, {});
                return;
            }
            for (auto& sub : duplicate_subs_)
            {
                if (std::count(deleted_ids_.begin(), deleted_ids_.end(), sub.id) > 0)
                {
                    hub->transport_log->deduplicate_unsubscribe_finished(ctx, sub);
                    deleted_subs_.push_back(sub);
                }
                else
                {
                    hub->transport_log->deduplicate_unsubscribe_conflict(ctx, sub);
                }
            }
            handler(ec, deleted_subs_);
        }
    }

    // List handler.
    void operator()(
        yplatform::yield_context<deduplicate_coro> yield_ctx,
        boost::system::error_code ec,
        sub_list subs)
    {
        subs_ = std::move(subs);
        operator()(yield_ctx, ec);
    }

    // Batch unsubscribe handler.
    void operator()(
        yplatform::yield_context<deduplicate_coro> yield_ctx,
        boost::system::error_code ec,
        std::vector<string> ids)
    {
        deleted_ids_ = std::move(ids);
        operator()(yield_ctx, ec);
    }

    void erase_non_mobile_subs()
    {
        subs_.erase(
            std::remove_if(
                subs_.begin(), subs_.end(), [](const sub_t& sub) { return sub.platform.empty(); }),
            subs_.end());
    }

    bool marked_as_duplicate(const sub_t& sub)
    {
        return std::count_if(
                   duplicate_subs_.begin(), duplicate_subs_.end(), [&sub](const sub_t& sub2) {
                       return sub.id == sub2.id;
                   }) > 0;
    }

    void mark_as_duplicate(const sub_t& sub, const sub_t& original, string& reason)
    {
        duplicate_subs_.push_back(sub);
        // We do not request uid and service from xtable,
        // since we already know them.
        // But we want to see uid and service in the log.
        auto& duplicate = duplicate_subs_.back();
        duplicate.uid = uid;
        duplicate.service = service;
        hub->transport_log->subscription_is_duplicate(ctx, duplicate, original, reason);
    }

    void mark_duplicates(const sub_t& sub, sub_list& subs)
    {
        for (auto& candidate_sub : subs)
        {
            if (candidate_sub.init_time >= sub.init_time)
            {
                // Definitely not a duplicate.
                continue;
            }
            auto matching_criterion =
                std::find_if(criteria_.begin(), criteria_.end(), [&](const criterion& c) {
                    return c.match(sub, candidate_sub);
                });
            if (matching_criterion != criteria_.end())
            {
                mark_as_duplicate(candidate_sub, sub, matching_criterion->reason);
            }
        }
    }

    std::vector<string> get_ids(const sub_list& subs)
    {
        std::vector<string> ret(subs.size());
        std::transform(subs.begin(), subs.end(), ret.begin(), [](const sub_t& s) { return s.id; });
        return ret;
    }

    struct criterion
    {
        std::function<bool(const sub_t&, const sub_t&)> match;
        string reason;
    };

    // Making these actually private would screw up {}-initialization.
    sub_list subs_{};
    sub_list duplicate_subs_{};
    std::vector<string> deleted_ids_{};
    sub_list deleted_subs_{};
    std::vector<criterion> criteria_{ { push_tokens_match, "push tokens match" },
                                      { devices_appnames_match, "devices and appnames match" } };
};

struct subscribe_handle_result : public api_coroutine
{
    std::shared_ptr<state> hub;
    expirable_stream_ptr stream;
    unsigned history_count;
    bool create_xtask;

    void operator()(const error_code& ec = {}, const sub_t& subscription = {})
    {
        reenter(this)
        {
            if (ec)
            {
                hub->transport_log->subscribe_failed(
                    stream->ctx(), subscription, message_for_error(ec));
                WEB_RESPONSE_CODE(stream, http_code_for_error(ec), message_for_error(ec));
            }
            else
            {
                hub->transport_log->subscribe_finished(stream->ctx(), subscription);
                if (stream->ctx()->is_cancelled()) return;
                if (auto http_stream = stream->detach_stream())
                {
                    http_stream->set_code(ymod_webserver::codes::ok);
                    if (subscription.init_local_id)
                    {
                        http_stream->add_header(
                            "X-Xiva-Position", std::to_string(subscription.init_local_id));
                        http_stream->add_header("X-Xiva-Count", std::to_string(history_count));
                    }
                    http_stream->result_body(subscription.id);
                }

                if (callback_uri::is_mobile_uri(subscription.callback_url) ||
                    callback_uri::is_apns_queue_uri(subscription.callback_url))
                {
                    if (hub->stats.deduplicate_xtable)
                    {
                        auto noop_handler = [](boost::system::error_code, sub_list) {};
                        yplatform::spawn(std::make_shared<deduplicate_coro>(
                            deduplicate_coro{ hub,
                                              stream->ctx(),
                                              subscription.uid,
                                              subscription.service,
                                              noop_handler }));
                    }
                    else
                    {
                        hub->transport_log->deduplication_disabled(
                            stream->ctx(), subscription.uid, subscription.service);
                    }
                }

                if (!hub->stats.convey_enabled)
                {
                    hub->transport_log->subscribe_convey_disabled(stream->ctx(), subscription);
                    return;
                }

                if (!hub->stats.robust_delivery) return;

                // Optimization: nothing to send yet - no need to create task.
                if (!create_xtask || subscription.init_local_id == 0) return;

                yield
                {
                    stream->context()->profilers.push("xtasks::create");
                    auto delay_flags = ymod_xtasks::delay_flags::wakeup_on_create |
                        ymod_xtasks::delay_flags::ignore_if_pending;
                    ymod_xtasks::task_draft draft = { subscription.uid,
                                                      subscription.service,
                                                      subscription.init_local_id,
                                                      "",
                                                      delay_flags };
                    hub->xtasks->create_task(stream->context(), draft, *this);
                }
                stream->context()->profilers.pop("xtasks::create");
                if (ec)
                {
                    hub->transport_log->subscribe_task_create_failed(
                        stream->ctx(), subscription, ec.message());
                    // collect errors, observer timer handler will analyse it later
                    if (hub->stats.control_leader)
                    {
                        hub->stats.xtasks_errors++;
                    }
                    return;

                    hub->transport_log->subscribe_task_created(stream->ctx(), subscription);
                }
            }
        }
    }

    // xtasks.create handler
    void operator()(const ymod_xtasks::error& err)
    {
        (*this)(err.code);
    }
};
}

struct subscribe : public api_coroutine
{
    std::shared_ptr<state> hub;
    expirable_stream_ptr stream;
    string uid;
    string service;
    string callback;
    string client;
    string session_key;
    string filter;
    string extra_data;
    ttl_t ttl;
    string uniq_id;
    string bb_connection_id;
    string uidset;
    local_id_t position;
    unsigned history_count;
    bool strict_position;
    sub_t subscription = {};

    void operator()(
        const ymod_xstore::error& err = {},
        const ymod_xstore::range_counters_list& counters_list = {})
    {
        time_t begin_ts = 0;
        ::yxiva::filter_set filter_set;

        reenter(this)
        {
            if (auto filter_parse_result = filter::parse(filter_set, filter); !filter_parse_result)
            {
                WEB_RESPONSE_LOG_G(
                    error, stream, bad_request, "bad filter: " + filter_parse_result.error_reason);
                return;
            }

            subscription.uid = uid;
            subscription.service = service;
            subscription.filter = filter_set.to_string();
            subscription.callback_url = callback;
            subscription.extra_data = extra_data;
            subscription.client = client;
            subscription.session_key = session_key;
            subscription.ttl = ttl;
            subscription.init_local_id = position;
            subscription.ack_local_id = position;
            subscription.id = uniq_id;
            subscription.bb_connection_id = bb_connection_id;
            subscription.uidset = uidset;

            begin_ts = std::time(nullptr) - hub->settings->subscribe_counters_lookup_time;

            if (history_count)
            {
                yield hub->xstore->get_range_counters(
                    stream->ctx(),
                    user_id(uid),
                    { service },
                    begin_ts,
                    position,
                    history_count,
                    *this);
                static const ymod_xstore::range_counters EMPTY;

                if (err)
                {
                    WEB_RESPONSE_LOG_G(info, stream, internal_server_error, err.code.message());
                    return;
                }
                if (counters_list.size() > 1)
                {
                    WEB_RESPONSE_LOG_G(
                        info, stream, internal_server_error, "unexpected range counters list size");
                    return;
                }
                if (counters_list.size() > 0 && counters_list[0].service != subscription.service)
                {
                    WEB_RESPONSE_LOG_G(
                        info,
                        stream,
                        internal_server_error,
                        "unexpected range counters list service");
                    return;
                }
                auto& counters = counters_list.size() ? counters_list[0] : EMPTY;

                auto position = subscription.init_local_id;
                detail::correct_range(counters.bottom, counters.top, position, history_count);

                bool position_changed =
                    subscription.init_local_id != 0 && position != subscription.init_local_id;
                bool create_xtask = true;
                if ((history_count == 0 || position_changed) && strict_position)
                {
                    create_xtask = false;
                    subscription.init_local_id = counters.top;
                    history_count = 0;
                }
                else
                {
                    subscription.init_local_id = position;
                }

                subscription.ack_local_id = subscription.init_local_id;

                detail::subscribe_handle_result cb = {
                    {}, hub, stream, history_count, create_xtask
                };
                hub->xtable->subscribe(stream->ctx(), subscription, cb);
            }
            else
            {
                detail::subscribe_handle_result cb = { {}, hub, stream, 0, false };
                hub->xtable->subscribe(stream->ctx(), subscription, cb);
            }
        }
    }
};

struct subscribe_mobile : public api_coroutine
{
    std::shared_ptr<state> hub;
    expirable_stream_ptr stream;
    string uid;
    string service;
    string callback;
    string client;
    string device_uuid;
    string filter;
    string extra_data;
    ttl_t ttl;
    string uniq_id;
    local_id_t position;
    string platform;
    string device;
    string bb_connection_id;

    void operator()()
    {
        ::yxiva::filter_set filter_set;
        auto filter_parse_result = filter::parse(filter_set, filter);
        if (!filter_parse_result)
        {
            WEB_RESPONSE_LOG_G(
                error, stream, bad_request, "bad filter: " + filter_parse_result.error_reason);
            return;
        }
        detail::subscribe_handle_result cb = { {}, hub, stream, 0, true };
        hub->xtable->subscribe_mobile(
            stream->ctx(),
            uid,
            service,
            filter_set.to_string(),
            callback,
            extra_data,
            client,
            device_uuid,
            ttl,
            uniq_id,
            position,
            platform,
            device,
            bb_connection_id,
            cb);
    }
};

struct deduplicate : public api_coroutine
{
    std::shared_ptr<state> hub;
    stream_ptr stream;
    string uid;
    string service;

    void operator()(
        boost::system::error_code ec = boost::system::error_code{},
        sub_list deleted_subs = sub_list{})
    {
        reenter(this)
        {
            yield yplatform::spawn(std::make_shared<detail::deduplicate_coro>(
                detail::deduplicate_coro{ hub, stream->ctx(), uid, service, *this }));
            if (ec)
            {
                WEB_RESPONSE_CODE(stream, http_code_for_error(ec), message_for_error(ec));
            }
            else
            {
                stream->result(ymod_webserver::codes::ok, serialize_subs(deleted_subs));
            }
        }
    }

    string serialize_subs(const sub_list& subs)
    {
        json_value result;
        auto&& unsubscribed_array = result["unsubscribed"];
        unsubscribed_array.set_array();
        for (auto& sub : subs)
        {
            json_value item;
            item["session_key"] = sub.session_key;
            unsubscribed_array.push_back(item);
        }
        return json_write_styled(result);
    }
};
}}}
