#pragma once

#include "common.h"

namespace yxiva { namespace hub { namespace api {

struct unsubscribe : public api_coroutine
{
    std::shared_ptr<state> hub;
    expirable_stream_ptr stream;
    string uid;
    string service;
    string subscription_id;

    void operator()(const error_code& err = {}, const string& /*subscribed_id*/ = {})
    {
        reenter(this)
        {
            yield hub->xtable->unsubscribe(stream->context(), uid, service, subscription_id, *this);

            if (err)
            {
                hub->transport_log->unsubscribe_failed(
                    stream->ctx(), uid, service, subscription_id, err.message());
                WEB_RESPONSE_CODE_LOG_G(
                    error, stream, http_code_for_error(err), message_for_error(err));
            }
            else
            {
                hub->transport_log->unsubscribe_finished(
                    stream->ctx(), uid, service, subscription_id);
                WEB_RESPONSE(stream, ok, "");
            }
        }
    }
};

struct unsubscribe_mobile : public api_coroutine
{
    std::shared_ptr<state> hub;
    expirable_stream_ptr stream;
    string uid;
    string service;
    string device_uuid;

    void operator()(const error_code& err = {}, const string& unsubscribed_id = {})
    {
        reenter(this)
        {
            yield hub->xtable->unsubscribe_mobile(
                stream->context(), uid, service, device_uuid, *this);

            if (err)
            {
                hub->transport_log->unsubscribe_failed(
                    stream->ctx(), uid, service, unsubscribed_id, err.message());
                WEB_RESPONSE_CODE_LOG_G(
                    error, stream, http_code_for_error(err), message_for_error(err));
            }
            else
            {
                hub->transport_log->unsubscribe_finished(
                    stream->ctx(), uid, service, unsubscribed_id);
                WEB_RESPONSE(stream, ok, "");
            }
        }
    }
};

struct batch_unsubscribe : public api_coroutine
{
    std::shared_ptr<state> hub;
    expirable_stream_ptr stream;
    string uid;
    string service;
    std::vector<string> subscription_ids;
    std::time_t init_not_after;

    void operator()(const error_code& ec = {}, const std::vector<string>& unsubscribed_ids = {})
    {
        reenter(this)
        {
            if (subscription_ids.empty())
            {
                WEB_RESPONSE_CODE_LOG_G(
                    error, stream, ymod_webserver::codes::bad_request, "empty ids list");
                return;
            }
            if (init_not_after == 0) init_not_after = std::time(nullptr);

            yield hub->xtable->batch_unsubscribe(
                stream->ctx(), uid, service, subscription_ids, init_not_after, *this);
            if (ec)
            {
                WEB_RESPONSE_CODE_LOG_G(
                    error, stream, http_code_for_error(ec), message_for_error(ec));
                // On failed batch_unsubscribe attempt report fail for all
                // requested ids.
                for (auto& subscription_id : subscription_ids)
                {
                    hub->transport_log->unsubscribe_failed(
                        stream->ctx(), uid, service, subscription_id, ec.message());
                }
            }
            else
            {
                WEB_RESPONSE(stream, ok, "");
                // In case of success only log actually unsubscribed ids.
                for (auto& unsubscribed_id : unsubscribed_ids)
                {
                    hub->transport_log->unsubscribe_finished(
                        stream->ctx(), uid, service, unsubscribed_id);
                }
            }

            YLOG_CTX_GLOBAL(stream->ctx(), info)
                << "requested batch_unsubscribe for ids: ["
                << boost::algorithm::join(subscription_ids, ", ") << "]";
        }
    }
};
}}}
