#pragma once

#include "common.h"
#include "cluster_metadata.h"

namespace yxiva { namespace hub { namespace api {

namespace {

bool verify_topology_diff(
    const std::set<peer_info> old_topology,
    const std::set<peer_info>& new_topology)
{
    std::vector<peer_info> added_peers;
    std::vector<peer_info> deleted_peers;
    std::set_difference(
        new_topology.begin(),
        new_topology.end(),
        old_topology.begin(),
        old_topology.end(),
        std::back_inserter(added_peers));
    std::set_difference(
        old_topology.begin(),
        old_topology.end(),
        new_topology.begin(),
        new_topology.end(),
        std::back_inserter(deleted_peers));
    return added_peers.size() == deleted_peers.size() && added_peers.size() == 1;
}

}

struct update_topology
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;
    string peers_str;
    bool force_update;

    void operator()()
    {
        auto controller = hub->controller.lock();
        if (!controller || !hub->stats.control_leader)
        {
            YLOG_CTX_GLOBAL(stream->ctx(), info) << "not a leader";
            stream->result(codes::forbidden, "not a leader");
            return;
        }

        std::set<peer_info> peers;
        try
        {
            auto parsed = json_parse(peers_str, json_type::tarray);
            for (size_t i = 0; i < parsed.size(); ++i)
            {
                [[maybe_unused]] auto [it, inserted] =
                    peers.insert(peer_info::from_string(parsed[size_t(i)].to_string()));
                if (!inserted) throw std::domain_error("peer duplicate");
            }
        }
        catch (std::exception& e)
        {
            stream->result(codes::bad_request, e.what());
            return;
        }

        auto force_update_handler = [stream = stream](auto& /* a */, auto & /* b */) -> bool {
            stream->result(codes::ok, "ok");
            return true;
        };

        auto update_handler = [stream = stream](auto& a, auto& b) -> bool {
            if (!verify_topology_diff(a, b))
            {
                YLOG_CTX_GLOBAL(stream->ctx(), info) << "inacceptable topology changes";
                stream->result(codes::bad_request, "inacceptable topology changes");
                return false;
            }
            else
            {
                stream->result(codes::ok, "ok");
                return true;
            }
        };

        // Can't use simple ternary operator for second argument
        // because of different lambda types.
        force_update ? controller->async_set_topology(peers, force_update_handler) :
                       controller->async_set_topology(peers, update_handler);
    }
};
}}}
