#pragma once

#include "common.h"

namespace yxiva { namespace hub { namespace api { namespace xtasks {

struct alive : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;
    string worker;

    void operator()(const ymod_xtasks::error& error = {})
    {
        reenter(this)
        {
            yield hub->xtasks->alive(stream->ctx(), worker, *this);
            if (error)
            {
                json_value json;
                json["error"] = error.code.message();
                json["error-ext"] = error.ext_reason;
                stream->result(ymod_webserver::codes::expectation_failed, json_write(json));
            }
            else
            {
                stream->result(ymod_webserver::codes::ok, "");
            }
        }
    }
};

struct cleanup_active : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;
    time_t deadline_sec;

    void operator()(const ymod_xtasks::error& error = {}, size_t count = {})
    {
        reenter(this)
        {
            yield hub->xtasks->cleanup_active(stream->ctx(), deadline_sec, *this);
            if (error)
            {
                json_value json;
                json["error"] = error.code.message();
                json["error-ext"] = error.ext_reason;
                stream->result(ymod_webserver::codes::expectation_failed, json_write(json));
            }
            else
            {
                json_value json;
                json["count"] = count;
                stream->result(ymod_webserver::codes::ok, json_write(json));
            }
        }
    }
};

struct wakeup_delayed : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;

    void operator()(const ymod_xtasks::error& error = {}, size_t count = {})
    {
        reenter(this)
        {
            yield hub->xtasks->wakeup_delayed(stream->ctx(), *this);
            if (error)
            {
                json_value json;
                json["error"] = error.code.message();
                json["error-ext"] = error.ext_reason;
                stream->result(ymod_webserver::codes::expectation_failed, json_write(json));
            }
            else
            {
                json_value json;
                json["count"] = count;
                stream->result(ymod_webserver::codes::ok, json_write(json));
            }
        }
    }
};

struct cleanup_workers : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;
    time_t deadline_sec;

    void operator()(const ymod_xtasks::error& error = {}, size_t count = {})
    {
        reenter(this)
        {
            yield hub->xtasks->cleanup_workers(stream->ctx(), deadline_sec, *this);
            if (error)
            {
                json_value json;
                json["error"] = error.code.message();
                json["error-ext"] = error.ext_reason;
                stream->result(ymod_webserver::codes::expectation_failed, json_write(json));
            }
            else
            {
                json_value json;
                json["count"] = count;
                stream->result(ymod_webserver::codes::ok, json_write(json));
            }
        }
    }
};

struct clear : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;

    void operator()(const ymod_xtasks::error& error = {}, size_t count = {})
    {
        reenter(this)
        {
            yield hub->xtasks->clear(stream->ctx(), *this);
            if (error)
            {
                json_value json;
                json["error"] = error.code.message();
                json["error-ext"] = error.ext_reason;
                stream->result(ymod_webserver::codes::expectation_failed, json_write(json));
            }
            else
            {
                json_value json;
                json["count"] = count;
                stream->result(ymod_webserver::codes::ok, json_write(json));
            }
        }
    }
};

struct counters : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;

    void operator()(
        const ymod_xtasks::error& error = {},
        const ymod_xtasks::all_counters_result& result = {})
    {
        reenter(this)
        {
            yield hub->xtasks->get_all_counters(stream->ctx(), *this);
            if (error)
            {
                json_value json;
                json["error"] = error.code.message();
                json["error-ext"] = error.ext_reason;
                stream->result(ymod_webserver::codes::expectation_failed, json_write(json));
            }
            else
            {
                json_value json;
                json["total"] = result.total;
                json["pending"] = result.pending;
                json["active"] = result.active;
                json["delayed"] = result.delayed;
                json["workers"] = result.workers;
                stream->result(ymod_webserver::codes::ok, json_write(json));
            }
        }
    }
};

struct create_task : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;
    string uid;
    string service;
    local_id_t local_id;
    bool wakeup_on_create;
    bool ignore_delay_if_pending;

    void operator()(const ymod_xtasks::error& error = {})
    {
        reenter(this)
        {
            yield
            {
                ymod_xtasks::delay_flags_t delay_flags = ymod_xtasks::delay_flags::none;
                if (wakeup_on_create) delay_flags |= ymod_xtasks::delay_flags::wakeup_on_create;
                if (ignore_delay_if_pending)
                    delay_flags |= ymod_xtasks::delay_flags::ignore_if_pending;
                ymod_xtasks::task_draft draft;
                draft.uid = uid;
                draft.service = service;
                draft.local_id = boost::lexical_cast<ymod_xtasks::local_id_t>(local_id);
                draft.delay_flags = delay_flags;
                hub->xtasks->create_task(stream->ctx(), draft, *this);
            }

            if (error)
            {
                json_value json;
                json["error"] = error.code.message();
                json["error-ext"] = error.ext_reason;
                stream->result(ymod_webserver::codes::expectation_failed, json_write(json));
            }
            else
            {
                stream->result(ymod_webserver::codes::ok, "OK");
            }
        }
    }
};

struct delay_task : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;
    string worker;
    string id;
    time_t delay_sec;

    void operator()(const ymod_xtasks::error& error = {})
    {
        reenter(this)
        {
            yield hub->xtasks->delay_task(stream->ctx(), worker, id, delay_sec, *this);
            if (error)
            {
                json_value json;
                json["error"] = error.code.message();
                json["error-ext"] = error.ext_reason;
                stream->result(ymod_webserver::codes::expectation_failed, json_write(json));
            }
            else
            {
                stream->result(ymod_webserver::codes::ok, "");
            }
        }
    }
};

struct delay_task_shallow : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;
    string worker;
    string id;
    time_t delay_sec;

    void operator()(const ymod_xtasks::error& error = {})
    {
        reenter(this)
        {
            yield hub->xtasks->delay_task_shallow(stream->ctx(), worker, id, delay_sec, *this);
            if (error)
            {
                json_value json;
                json["error"] = error.code.message();
                json["error-ext"] = error.ext_reason;
                stream->result(ymod_webserver::codes::expectation_failed, json_write(json));
            }
            else
            {
                stream->result(ymod_webserver::codes::ok, "");
            }
        }
    }
};

struct fin_task : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;
    string worker;
    string id;

    void operator()(const ymod_xtasks::error& error = {})
    {
        reenter(this)
        {
            yield hub->xtasks->fin_task(stream->ctx(), worker, id, *this);
            if (error)
            {
                json_value json;
                json["error"] = error.code.message();
                json["error-ext"] = error.ext_reason;
                stream->result(ymod_webserver::codes::expectation_failed, json_write(json));
            }
            else
            {
                stream->result(ymod_webserver::codes::ok, "");
            }
        }
    }
};

struct get_summary : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;

    void operator()()
    {
        auto summary = hub->xtasks->get_summary_json();
        stream->result(ymod_webserver::codes::ok, summary);
    }
};

struct get_tasks : public api_coroutine
{
    std::shared_ptr<state> hub;
    ymod_webserver::http::stream_ptr stream;
    string worker;
    unsigned count;

    void operator()(
        const ymod_xtasks::error& error = {},
        const std::vector<ymod_xtasks::task>& tasks = {})
    {
        reenter(this)
        {
            yield hub->xtasks->get_tasks(stream->ctx(), worker, count, *this);
            if (error)
            {
                json_value json;
                json["error"] = error.code.message();
                json["error-ext"] = error.ext_reason;
                stream->result(ymod_webserver::codes::expectation_failed, json_write(json));
            }
            else
            {
                json_value json(json_type::tarray);
                for (auto cur = tasks.begin(); cur != tasks.end(); ++cur)
                {
                    json_value item;
                    item["id"] = cur->id;
                    item["uid"] = cur->uid;
                    item["service"] = cur->service;
                    json.push_back(item);
                }
                stream->result(ymod_webserver::codes::ok, json_write(json));
            }
        }
    }
};
}}}}