#pragma once

#include "peer_info.h"
#include <yxiva/core/types.h>
#include <yxiva/core/json.h>
#include <boost/property_tree/json_parser.hpp>

namespace yxiva { namespace hub {

struct cluster_metadata
{
    bool valid = false;
    bool robust_delivery = false;
    std::set<peer_info> peers; // cluster topology
    bool static_topology = false;
    string master_address;

    bool in_peers(const string& address) const
    {
        // Hack for local configurations.
        if (static_topology && peers.size() && peers.begin()->is_local()) return true;

        // Lookup with fake peer id.
        return peers.count({ 0, address });
    }

    unsigned peer_id(const string& address) const
    {
        // Lookup with fake peer id.
        if (auto it = peers.find({ 0, address }); it != peers.end())
        {
            return it->id;
        }
        else
        {
            // Hack for local configurations.
            if (static_topology && peers.size() && peers.begin()->is_local()) return 0;
            throw std::runtime_error("peer not found");
        }
    }

    void load(const string& src)
    {
        valid = false;
        auto parsed = json_parse(src);
        robust_delivery = parsed["robust"].to_bool();
        if (!static_topology)
        {
            peers.clear();
            for (size_t i = 0; i < parsed["peers"].size(); ++i)
            {
                peers.insert(peer_info::from_string(parsed["peers"][size_t(i)].to_string()));
            }
            if (peers.empty())
            {
                throw std::domain_error("invalid cluster config (peers)");
            }
        }
        valid = true;
    }

    string dump() const
    {
        if (!valid) return "";
        json_value result;
        result["robust"] = robust_delivery;
        result["peers"].set_array();
        for (auto& peer : peers)
        {
            result["peers"].push_back(peer.to_string());
        }
        return json_write(result);
    }

    string dump_topology() const
    {
        if (!valid) return "";
        json_value result(json_type::tarray);
        for (auto& peer : peers)
        {
            result.push_back(peer.to_string());
        }
        return json_write(result);
    }

    // XXX duplicates load() because of different YML and JSON
    // data representation in ptree.
    void parse_ptree(const yplatform::ptree& conf)
    {
        valid = false;
        robust_delivery = conf.get<bool>("robust");
        if (!static_topology)
        {
            peers.clear();
            yplatform::read_ptree(peers, conf, "peers");
            if (peers.empty())
            {
                throw std::domain_error("invalid cluster config (peers)");
            }
        }
        valid = true;
    }
};

}}
