#include "control_module.h"

#include <sintimers/queue.h>
#include <yplatform/util/shared_ptr_cast.h>

namespace yxiva { namespace hub {

namespace ph = std::placeholders;

control_module::control_module(yplatform::reactor& reactor, std::shared_ptr<state> state)
    : strand_(*reactor.io()), state_(state)
{
    timers_ = boost::make_shared<timers::queue_on_strand>(&strand_);
    observe_timer_ = timers_->create_timer();
    my_address_ = state_->netch->my_address();
}

void control_module::init()
{
    if (state_->settings->force_weak_delivery)
    {
        state_->stats.convey_enabled = true;
        state_->stats.robust_delivery = false;
        return;
    }

    auto self = yplatform::shared_from(this);
    state_->leasemeta->bind(
        state_->settings->cluster.name,
        std::bind(&control_module::handle_lease, self, ph::_2, ph::_4),
        std::bind(&control_module::handle_lease, self, "", ""));
    state_->leasemeta->start_acquire_lease(state_->settings->cluster.name);

    if (state_->settings->cluster.use_static_metadata)
    {
        cluster_metadata_ = state_->settings->cluster.static_metadata;
        cluster_metadata_.static_topology = true;
        reconfigure_network();
    }
}

void control_module::start()
{
    if (state_->settings->force_weak_delivery) return;
    observe_timer_start();
}

void control_module::stop()
{
    if (state_->settings->force_weak_delivery) return;
    state_->leasemeta->update_lease_time(state_->settings->cluster.name, milliseconds(200));
    std::this_thread::sleep_for(milliseconds(100));
    state_->leasemeta->stop_acquire_lease(state_->settings->cluster.name);
    observe_timer_stop();
}

void control_module::handle_lease(const ymod_lease::node_id& nodeId, const ymod_lease::value& value)
{
    if (nodeId.empty())
    {
        YLOG_L(error) << "cluster metadata unreachable, finalizing";
        if (state_->stats.control_leader)
        {
            state_->stats.control_leader = false;
            state_->xtasks_service->stop();
        }
        state_->stats.convey_enabled = false;
        state_->stats.finalize = true;
        return;
    }

    YLOG_L(info) << "cluster metadata leader=" << nodeId << " value=" << value;
    state_->stats.finalize = false;

    if (value.size())
    {
        try
        {
            cluster_metadata_.load(value);
        }
        catch (std::exception& e)
        {
            YLOG_L(error) << "invalid cluster metadata: " << e.what();
        }
    }

    cluster_metadata_.master_address = state_->leasemeta->address_from_id(nodeId);
    state_->stats.set_cluster_metadata(cluster_metadata_);

    if (cluster_metadata_.valid && !cluster_metadata_.in_peers(my_address_))
    {
        YLOG_L(notice) << "my address " << my_address_ << " is out of cluster, finalizing";
        state_->stats.finalize = true;
        return;
    }

    state_->stats.control_leader = state_->leasemeta->node_id() == nodeId;
    if (state_->stats.control_leader)
    {
        state_->xtasks_service->start();
        state_->stats.self_diag.reset();
    }
    else
    {
        state_->xtasks_service->stop();
    }

    state_->stats.convey_enabled = true;

    if (!cluster_metadata_.valid)
    {
        YLOG_L(warning) << "no cluster metadata, switching to weak delivery mode";
        state_->stats.robust_delivery = false;
        return;
    }

    state_->stats.robust_delivery = cluster_metadata_.robust_delivery;

    if (!cluster_metadata_.static_topology) reconfigure_network();
}

bool control_module::async_set_robust_delivery(bool enabled)
{
    if (state_->settings->force_weak_delivery) return false;
    if (!state_->stats.control_leader) return false;

    strand_.dispatch([this, enabled]() {
        if (!state_->stats.control_leader) return;
        if (!cluster_metadata_.valid) return;

        auto target = cluster_metadata_;
        target.robust_delivery = enabled;
        state_->leasemeta->update_acquire_value(state_->settings->cluster.name, target.dump());
    });
    return true;
}

bool control_module::async_set_topology(
    const std::set<peer_info>& topology,
    const topology_validator& validator)
{
    if (state_->settings->force_weak_delivery) return false;
    if (!state_->stats.control_leader) return false;

    strand_.dispatch([this, topology, validator]() {
        if (!state_->stats.control_leader) return;
        if (cluster_metadata_.static_topology) return;
        if (!validator(cluster_metadata_.peers, topology)) return;

        auto target = cluster_metadata_.valid ? cluster_metadata_ : cluster_metadata{};
        target.valid = true;
        target.peers = topology;

        YLOG_L(info) << "storing new metadata " << target.dump();
        state_->leasemeta->update_acquire_value(state_->settings->cluster.name, target.dump());
    });
    return true;
}

void control_module::reconfigure_network()
{
    auto connect_to_port_with_offset = [](auto&& netch, auto peers, auto port_offset) {
        std::set<string> netch_peers;
        for (auto& peer : peers)
        {
            auto hostinfo = ymod_messenger::make_host_info(peer.address);
            hostinfo.port += port_offset;
            netch_peers.insert(hostinfo.to_string());
        }
        netch->connect_to_cluster(netch_peers);
    };

    state_->paxos_node->set_id(cluster_metadata_.peer_id(my_address_));
    connect_to_port_with_offset(state_->netch, cluster_metadata_.peers, 0);
    connect_to_port_with_offset(
        state_->lease_netch, cluster_metadata_.peers, state_->settings->cluster.lease_port_offset);
    connect_to_port_with_offset(
        state_->sync_netch, cluster_metadata_.peers, state_->settings->cluster.sync_port_offset);
}

void control_module::observe_timer_start()
{
    static const auto interval = seconds(1);
    observe_timer_->async_wait(interval, [this]() { handle_observe_timer(); });
}

void control_module::observe_timer_stop()
{
    observe_timer_->cancel();
}

void control_module::handle_observe_timer()
{
    sample_xtasks();
    bool automatic_switch = state_->stats.control_leader &&
        !state_->stats.manual_xtasks_degradation && !state_->stats.manual_xstore_degradation;
    if (state_->stats.robust_delivery)
    {
        update_xtasks_parameters();
        if (automatic_switch)
        {
            try_disable_robust_delivery();
        }
    }
    else
    {
        state_->stats.ignore_xstore_errors = state_->xstore->in_fallback();
        if (automatic_switch)
        {
            try_enable_robust_delivery();
        }
    }
    observe_timer_start();
}

void control_module::try_enable_robust_delivery()
{
    if (state_->stats.manual_xtasks_degradation) return;
    if (state_->stats.manual_xstore_degradation) return;
    if (state_->xstore->in_fallback()) return;
    if (count_xtasks_bad_samples() == 0)
    {
        YLOG_L(notice) << "switching to robust delivery";
        async_set_robust_delivery(true);
    }
}

void control_module::try_disable_robust_delivery()
{
    bool xstore_in_fallback = state_->xstore->in_fallback();
    auto xtasks_broken = count_xtasks_bad_samples() == state_->stats.self_diag.samples().size();
    if (xstore_in_fallback || xtasks_broken)
    {
        YLOG_L(notice) << "fallback to weak delivery: xtasks_samples="
                       << state_->stats.self_diag.dump()
                       << " xstore_in_fallback=" << xstore_in_fallback;
        async_set_robust_delivery(false);
    }
}

void control_module::update_xtasks_parameters()
{
    auto paxos_status = state_->paxos_node->get_status();
    if (paxos_status.master_available)
    {
        state_->xtasks_worker->lazy_mode(paxos_status.is_master);
    }
}

void control_module::sample_xtasks()
{
    auto paxos_status = state_->paxos_node->get_status();
    diag_sample sample;
    sample.errors = state_->stats.xtasks_errors;
    sample.no_master = !paxos_status.master_available;
    state_->stats.xtasks_errors -= sample.errors;
    state_->stats.self_diag.add(sample);
}

unsigned control_module::count_xtasks_bad_samples()
{
    unsigned bad_samples = 0;
    for (auto& sample : state_->stats.self_diag.samples())
    {
        if (sample.errors || sample.no_master)
        {
            bad_samples++;
        }
    }
    return bad_samples;
}

}}
