#include "apns_queue_gate.h"

#include <yxiva/core/platforms.h>
#include <yxiva/core/packing.hpp>
#include <ymod_httpclient/url_encode.h>
#include <algorithm>

namespace yxiva { namespace hub {

namespace p = std::placeholders;

apns_queue_gate::apns_queue_gate(
    yplatform::reactor& reactor,
    const apns_queue_gate_settings& settings)
    : settings_(settings), http_client_(reactor, settings_.http)
{
}

void apns_queue_gate::send_message(const packet& packet, const handler_t& handler)
{
    auto task = std::make_shared<apns_queue_gate::task>(packet, handler);
    // This can only happen in case of error or manually added apns queue subscription.
    if (task->sub.platform != platform::APNS || task->sub.app_name.empty() ||
        task->sub.device.empty())
    {
        const char* error = task->sub.platform != platform::APNS ?
            "invalid queue subscription platform" :
            (task->sub.app_name.empty() ? "queue subscription with empty app name" :
                                          "queue subscription with empty device id");
        YLOG_CTX_GLOBAL(packet.ctx, error) << error;
        return task->result(gate_result::unsubscribe, error);
    }
    // Try to repack message to ensure it's correct before making any requests.
    if (auto repack_result = repack_message_if_needed(packet, task->sub, task->requests))
    {
        if (task->requests.empty())
        {
            YLOG_CTX_GLOBAL(packet.ctx, info) << "no notifications generated for "
                                                 "subscription_id="
                                              << task->sub.id;
            return task->result(gate_result::ignored, "no notifications generated");
        }
        // Check impl subscription exists to implement gate unsubscribe behavior.
        http_client_.async_run(
            packet.ctx,
            prepare_list(task->sub),
            settings_.http_options,
            std::bind(&apns_queue_gate::handle_list, this, task, p::_1, p::_2));
    }
    else
    {
        YLOG_CTX_GLOBAL(packet.ctx, info)
            << "repack failed with error=\"" << repack_result.error_reason << "\"";
        task->result(gate_result::unprocessable, repack_result.error_reason);
    }
}

void apns_queue_gate::handle_list(
    std::shared_ptr<task> task,
    const boost::system::error_code& ec,
    yhttp::response response)
{
    auto device_matcher = [&task](const json_value& sub) {
        return json_get<string>(sub, "device", string()) == task->sub.device;
    };
    auto& packet = task->packet;

    if (ec || response.status != 200)
    {
        YLOG_CTX_GLOBAL(packet.ctx, error)
            << "hub list failed: "
            << (ec ? ("error " + ec.message()) :
                     ("response " + std::to_string(response.status) + " " + response.body));
        return task->result(gate_result::fail, "failed to list subscriptions");
    }
    json_value list;
    if (auto res = json_parse(list, response.body, json_type::tarray))
    {
        if (std::none_of(list.array_begin(), list.array_end(), device_matcher))
        {
            return task->result(gate_result::unsubscribe, "no subscription");
        }
        do_send(task);
    }
    else
    {
        YLOG_CTX_GLOBAL(packet.ctx, error) << "bad list response: " << res.error_reason;
        task->result(gate_result::fail, "failed to list subscriptions");
    }
}

void apns_queue_gate::do_send(std::shared_ptr<task> task)
{
    auto& packet = task->packet;
    message message;
    bool bright = packet.bright && task->requests.front().is_bright() && packet.ttl;
    auto res = prepare_message(task, message, bright);
    if (!res)
    {
        YLOG_CTX_GLOBAL(packet.ctx, info) << "invalid payload: " << res.error_reason;
        return task->result(gate_result::ignored, res.error_reason);
    }

    // Send silent pushes with /fast_binary_notify,
    // mobile gate will then forward 'ttl=0' for messages with zero ttl.
    const string& url = bright ? "/binary_notify/" : "/fast_binary_notify/";
    http_client_.async_run(
        packet.ctx,
        prepare_notify(task->sub, message, url),
        settings_.http_options,
        std::bind(&apns_queue_gate::handle_send, this, task, bright, p::_1, p::_2));
}

void apns_queue_gate::handle_send(
    std::shared_ptr<task> task,
    bool bright,
    const boost::system::error_code& ec,
    yhttp::response response)
{
    // bright -> send response then handle rest, if any
    if (ec || response.status != 200)
    {
        YLOG_CTX_GLOBAL(task->packet.ctx, error)
            << "failed to notify: "
            << (ec ? ("error " + ec.message()) :
                     ("response " + std::to_string(response.status) + " " + response.body));
        return task->result(gate_result::fail, "failed to notify");
    }
    if (bright || task->requests.empty())
    {
        task->result(gate_result::success, "");
    }
    if (!task->requests.empty())
    {
        do_send(task);
    }
}

yhttp::request apns_queue_gate::prepare_list(const apns_queue_sub_t& sub)
{
    return yhttp::request::GET(
        "/list_json" +
        ymod_httpclient::url_encode(
            { { "service", sub.service }, { "uid", sub.queue_id }, { "db_role", "master" } }));
}

operation::result apns_queue_gate::prepare_message(
    std::shared_ptr<task> task,
    message& message,
    bool bright)
{
    assert(!task->requests.empty());
    auto& packet = task->packet;
    auto& req = task->requests.pop_front();
    json_value payload;
    if (req.payload().size())
    {
        auto res = json_parse(payload, req.payload());
        if (!res) return res;
    }
    if (!settings_.suppress_apns_queue_repack)
    {
        auto&& xiva_obj = payload["xiva"];
        xiva_obj["svc"] = packet.service;
        xiva_obj["usr"] = packet.uid;
    }

    message.uid = task->sub.queue_id;
    message.service = task->sub.service;
    message.raw_data = json_write(payload);
    // Can probably skip this check for silent push.
    if (message.raw_data.size() > settings_.max_payload_size)
    {
        return "payload is too long";
    }
    message.transit_id = packet.message.transit_id;
    if (task->repack.size())
    {
        message.repacking_rules[platform::APNS_QUEUE] = task->repack;
    }
    message.ttl = bright ? packet.ttl : 0;
    message.bright = bright;
    return operation::success;
}

yhttp::request apns_queue_gate::prepare_notify(
    const apns_queue_sub_t& sub,
    const message& msg,
    string url)
{
    url.append(sub.service);
    return yhttp::request::POST(
        url + ymod_httpclient::url_encode({ { "uid", sub.queue_id }, { "priority", "high" } }),
        pack(msg));
}

}}
