#include "exhaust.h"

#include "hacks.h"
#include "fcm_batch_aggregator.h"
#include <yxiva/core/services/names.h>
#include <yxiva/core/filter.h>
#include <yxiva/core/callbacks.h>
#include <yxiva/core/platforms.h>
#include <set>
#include <string>

namespace yxiva { namespace hub {

namespace p = std::placeholders;

static const string EMPTY;

exhaust::exhaust(yplatform::reactor& reactor)
    : reactor_(yplatform::reactor::make_not_owning_copy(reactor))
{
}

void exhaust::init(const yplatform::ptree& conf)
{
    auto mod_httpclient = yplatform::find<yhttp::call>("http_client");
    xtable_ = yplatform::find<XTable>("xtable");
    mod_log_ = yplatform::find<mod_log>("xivahub_log");

    settings_.load(conf);

    // http gate needs to have maximum response length limited
    // as it also serves http callbacks that can have unpredictalbe responses
    http_gate_ = std::make_shared<http_gate>(*reactor_, settings_.http);
    ws_gate_ = std::make_shared<http_gate>(*reactor_, settings_.ws);
    mobile_gate_ = std::make_shared<mobile_gate>(*reactor_, settings_.mobile);
    webpush_gate_ = std::make_shared<webpush_gate>(*reactor_, settings_.webpush);
    apns_queue_gate_ = std::make_shared<apns_queue_gate>(*reactor_, settings_.apns_queue);

    pipe_settings batch_pipe_st(conf.get<size_t>("batch_processor_pipe.capacity"));
    batch_pipe_st.set_window(conf.get<size_t>("batch_processor_pipe.parallel"));
    batch_pipe_st.set_grouped_flush_size(conf.get<size_t>("batch_processor_pipe.flush_size"));
    auto reactor =
        yplatform::global_reactor_set->get(conf.get<string>("batch_processor_pipe.reactor"));
    send_proc_ = std::make_shared<batch_processor>(
        *reactor->io(), batch_pipe_st, std::bind(&exhaust::send, this, p::_1, p::_2, p::_3, p::_4));

    pipe_settings fcm_pipe_st(conf.get<size_t>("fcm_batch_processor_pipe.capacity"));
    fcm_pipe_st.set_window(conf.get<size_t>("fcm_batch_processor_pipe.parallel"));
    fcm_pipe_st.set_grouped_flush_size(conf.get<size_t>("fcm_batch_processor_pipe.flush_size"));
    auto fcm_reactor =
        yplatform::global_reactor_set->get(conf.get<string>("fcm_batch_processor_pipe.reactor"));
    fcm_proc_ = std::make_shared<fcm_batch_processor>(*fcm_reactor, fcm_pipe_st, settings_.mobile);
}

void exhaust::start()
{
    send_proc_->start();
    fcm_proc_->start();
}

void exhaust::fini()
{
    send_proc_->stop();
    fcm_proc_->stop();
}

yplatform::ptree exhaust::get_stats() const
{
    yplatform::ptree stats;
    auto batch_stream = send_proc_->input();
    auto fcm_batch_stream = fcm_proc_->input();
    stats.put("batch_processor.buffer_size", batch_stream->buffer_size());
    stats.put("batch_processor.commited_size", batch_stream->total_commited_size());
    stats.put("fcm_batch_processor.buffer_size", fcm_batch_stream->buffer_size());
    stats.put("fcm_batch_processor.commited_size", fcm_batch_stream->total_commited_size());
    return stats;
}

void exhaust::convey(task_context_ptr ctx, shared_ptr<message> message, const convey_callback_t& cb)
{
    XTable::find_options options;
    options.db_role = XTable::db_role::replica;
    xtable_->find(
        ctx,
        message->uid,
        message->service,
        options,
        boost::bind(&exhaust::handle_list, shared_from(this), ctx, message, _1, _2, cb));
}

void exhaust::handle_list(
    task_context_ptr initial_ctx,
    shared_ptr<message> message,
    const error_code& err,
    const sub_list& list,
    convey_callback_t& cb)
{
    if (err)
    {
        YLOG_CTX_LOCAL(initial_ctx, error) << "exhaust convey storage error: " << err.value();
        cb(make_error_code(err_code::err_code_storage_fail), EMPTY);
        return;
    }

    auto ctx = boost::make_shared<convey_context>(initial_ctx, message, list);
    cb(error_code(), EMPTY);
    convey(ctx, std::bind([]() {}));
}

void exhaust::convey(convey_context_ptr ctx, const report_callback_t& report_cb)
{
    auto& list = ctx->subscriptions;
    size_t index = 0;
    for (auto it = list.begin(); it != list.end(); ++it, ++index)
    {
        YLOG_CTX_LOCAL(ctx, debug) << "exhaust convey found in table: " << it->to_string();
        auto action = filter::parse_and_apply(*it, *ctx->message);

        if (action == filter::action::skip)
        {
            mod_log_->convey_skip_filtered(packet(ctx, *ctx->message, *it));
            if (report_cb)
            {
                report_cb(make_error_code(err_code_filtered), *it, EMPTY);
            }
            continue;
        }

        struct packet packet(ctx, *ctx->message, *it);
        if (action == filter::action::send_silent) packet.mark_silent_delivery();

        if (report_cb)
        {
            auto& sub = *it;
            send(packet, true, [&sub, report_cb](const error_code& code, const string& data) {
                report_cb(code, sub, data);
            });
        }
        else
        {
            send(packet, true, [](const error_code&, const string&) {});
        }
    }
}

void exhaust::send(
    const packet& packet,
    bool update,
    const convey_callback_t& cb,
    push_request_cache* cache)
{
    static const std::map<string, string> no_gate_fields;

    if (callback_uri::is_mobile_uri(packet.subscription.callback_url))
    {
        mobile_gate_->send_message(
            packet,
            cache,
            std::bind(&exhaust::handle_gate_send, shared_from(this), packet, p::_1, update, cb));
    }

    else if (callback_uri::is_webpush_uri(packet.subscription.callback_url))
    {
        webpush_gate_->send_message(
            packet,
            std::bind(&exhaust::handle_gate_send, shared_from(this), packet, p::_1, update, cb));
    }

    else if (callback_uri::is_apns_queue_uri(packet.subscription.callback_url))
    {
        apns_queue_gate_->send_message(
            packet,
            std::bind(&exhaust::handle_gate_send, shared_from(this), packet, p::_1, update, cb));
    }

    else if (callback_uri::is_xiva_websocket_uri(packet.subscription.callback_url))
    {
        ws_gate_->send_message(
            packet,
            std::bind(&exhaust::handle_gate_send, shared_from(this), packet, p::_1, update, cb));
    }

    else
    {
        http_gate_->send_message(
            packet,
            std::bind(&exhaust::handle_gate_send, shared_from(this), packet, p::_1, update, cb));
    }
}

const time_duration& exhaust::send_timeout_for(const sub_t& subscription) const
{
    if (callback_uri::is_mobile_uri(subscription.callback_url))
    {
        return mobile_gate_->send_timeout_for(subscription);
    }
    else if (callback_uri::is_webpush_uri(subscription.callback_url))
    {
        return webpush_gate_->send_timeout_for(subscription);
    }
    else
    {
        return http_gate_->send_timeout_for(subscription);
    }
}

void exhaust::handle_gate_send(
    const packet& packet,
    gate_response resp,
    bool update,
    const convey_callback_t& cb)
{
    string new_callback;
    if (auto it = resp.data.find("new_callback"); it != resp.data.end())
    {
        new_callback = it->second;
        resp.data.erase(it);
    }
    auto res = error_code_from_gate_result(resp.result);
    try
    {
        if (!res)
        {
            if (resp.result == gate_result::success)
            {
                mod_log_->convey_success(packet, resp.body, resp.data);
            }
            else
            {
                mod_log_->convey_ignored(packet, resp.body, resp.data);
            }
            if (packet.message.local_id != 0 && update)
            {
                // Reset subscription retry interval, because even if there
                // was a message to retry, we will not deliver it, because
                // ack_local_id of this subscription has progressed further.
                xtable_->update(
                    packet.ctx,
                    packet.uid,
                    packet.service,
                    packet.subscription.id,
                    packet.subscription.ack_local_id,
                    packet.message.local_id,
                    0,
                    packet.message.event_ts,
                    std::bind(
                        &exhaust::handle_update, shared_from(this), packet, p::_1, p::_2, p::_3));
            }
        }
        else if (res.value() == err_code_subscription_deactivated)
        {
            mod_log_->convey_deactivated(packet, resp.data);
            new_callback = callback_uri::inactive_uri(packet.subscription.callback_url);
        }
        else if (res.value() == err_code_subscription_dropped)
        {
            mod_log_->convey_dropped(packet, resp.body, resp.data);
            xtable_->unsubscribe_overlapped(
                packet.ctx,
                packet.uid,
                packet.service,
                packet.subscription.id,
                settings_.drop_overlap_sec,
                std::bind(&exhaust::handle_unsubscribe, shared_from(this), packet, p::_1));
        }
        else
        {
            mod_log_->convey_failed(packet, resp.body, resp.data);
        }
        if (new_callback.size())
        {
            xtable_->update_callback(
                packet.ctx,
                packet.uid,
                packet.service,
                packet.subscription.id,
                packet.subscription.callback_url,
                new_callback,
                std::bind(&exhaust::handle_callback_update, shared_from(this), packet, p::_1));
        }
    }
    catch (const std::exception& ex)
    {
        YLOG_CTX_LOCAL(packet.ctx, error) << "handle_gate_send exception: " << ex.what();
    }
    cb(res, resp.result == gate_result::fail ? "internal error" : resp.body);
}

void exhaust::handle_callback_update(const packet& packet, const error_code& error)
{
    if (error)
    {
        mod_log_->convey_callback_update_failed(packet, error.message());
    }
    else
    {
        mod_log_->convey_callback_update_success(packet);
    }
}

void exhaust::handle_unsubscribe(const packet& packet, const error_code& error)
{
    if (error)
    {
        mod_log_->convey_unsubscribe_failed(packet, error.message());
    }
    else
    {
        mod_log_->convey_unsubscribe_success(packet);
    }
}

void exhaust::handle_update(
    const packet& packet,
    const error_code& error,
    bool exists,
    local_id_t local_id)
{
    if (error)
    {
        mod_log_->convey_update_failed(packet, error.message());
    }
    else
    {
        if (exists && local_id != packet.message.local_id)
        {
            mod_log_->convey_update_failed(
                packet,
                "conflict server local_id " + std::to_string(local_id) + " != our local_id " +
                    std::to_string(packet.message.local_id));
        }
        else
        {
            mod_log_->convey_update_success(packet);
        }
    }
}

void exhaust::batch_convey(convey_context_ptr ctx, const report_callback_t& report_cb)
{
    fcm_batch_aggregator batcher(FCM_MAX_BATCH_SIZE);
    auto fcm_cb =
        [this, report_cb](const fcm_batch_data_ptr& data, size_t sub_index, gate_response resp) {
            struct packet packet(data->ctx, *data->ctx->message, *data->subscriptions[sub_index]);
            if (!data->bright) packet.mark_silent_delivery();
            handle_gate_send(
                packet,
                resp,
                false,
                std::bind(report_cb, p::_1, *data->subscriptions[sub_index], p::_2));
        };
    auto make_data = [&report_cb, &ctx](const sub_t& sub, bool bright) {
        return std::make_shared<send_data>(
            ctx, sub, bright, std::bind(report_cb, p::_1, std::ref(sub), p::_2));
    };
    auto& subs = ctx->subscriptions;
    auto& message = ctx->message;
    auto general_batch = std::make_shared<std::vector<data_ptr>>();
    general_batch->reserve(subs.size());
    filter::action action;
    filter_set filter_set;

    auto send_proc_input = send_proc_->input();
    auto fcm_proc_input = fcm_proc_->input();

    for (auto& sub : subs)
    {
        try
        {
            if (!send_proc_input || !fcm_proc_input)
            {
                report_cb(make_error_code(err_code_cancelled), sub, "");
                continue;
            }

            action = filter::parse_and_apply(sub, *message);
            if (action == filter::action::skip)
            {
                mod_log_->convey_skip_filtered(packet(ctx, *message, sub).mark_silent_delivery());
                report_cb(make_error_code(err_code_filtered), sub, "filtered");
                continue;
            }
            bool bright = (action == filter::action::send_bright);

            if (callback_uri::is_mobile_uri(sub.callback_url) && sub.platform == platform::FCM)
            {
                push_subscription_params mob_sub(sub);
                if (mob_sub.app_name.empty())
                {
                    report_cb(make_error_code(err_code_gate_fail), sub, "no app name");
                    continue;
                }
                // XIVA-1989 If push token is fcm topic,
                // RTEC-4595 or push is disk-repacked (payload becomes subscription-dependent),
                // use general_batch to process it correctly.
                if (is_fcm_topic(mob_sub.push_token) || mob_sub.service == "disk-json" ||
                    mob_sub.service == "disk-user-events")
                {
                    general_batch->push_back(make_data(sub, bright));
                    continue;
                }

                auto pack = packet(ctx, *message, sub);
                mob_sub.repack_features = repack_features(pack, mob_sub, settings_.mobile);

                push_requests_queue req;
                if (!bright) pack.mark_silent_delivery();
                auto res = repack_message_if_needed(pack, mob_sub, req, *ctx->cache);
                if (!res)
                {
                    report_cb(make_error_code(err_code_unprocessable), sub, res.error_reason);
                    continue;
                }
                if (hacks::need_dump_request(pack))
                {
                    hacks::dump_request(req, pack.message.transit_id);
                }
                batcher.add(sub, mob_sub, bright, req, ctx, fcm_cb);
            }
            else
            {
                general_batch->push_back(make_data(sub, bright));
            }
        }
        catch (const std::exception& ex)
        {
            report_cb(make_error_code(err_code_gate_fail), sub, ex.what());
        }
    }

    if (!send_proc_input || !fcm_proc_input) return;

    send_proc_input->put_range(general_batch, [](const batch_range<data_ptr>& data_range) {
        for (auto& data : *data_range)
        {
            data->cb(make_error_code(err_code_service_unavailable), EMPTY);
        }
    });
    fcm_proc_input->put_range(
        batcher.get_all(), [](const batch_range<fcm_batch_data_ptr>& data_range) {
            for (auto& data : *data_range)
            {
                for (size_t i = 0; i < data->subscriptions.size(); ++i)
                {
                    data->cb(data, i, { gate_result::service_unavailable, "", {} });
                }
            }
        });
}

}}
