#pragma once

#include <yxiva/core/types.h>
#include <yxiva/core/subscriptions.h>
#include <yxiva/core/json.h>
#include <map>

namespace yxiva { namespace hub {

enum class gate_result
{
    success,
    fail,
    ignored,
    unsubscribe,
    bad_request,
    rate_limit,
    push_service_fail,
    timeout,
    service_unavailable,
    forbidden,
    deactivate,
    unprocessable
};

struct gate_response
{
    gate_result result;
    string body;
    // Additional data that gate provides for logging and other uses.
    std::map<string, string> data;
};

enum class gates
{
    http,
    xiva_mobile
};

inline gate_result gate_result_from_http_code(int code)
{
    switch (code)
    {
    case 200:
        return gate_result::success;
    case 202:
        return gate_result::ignored;
    case 204:
        return gate_result::deactivate;
    case 205:
        return gate_result::unsubscribe;
    case 400:
        return gate_result::bad_request;
    case 403:
        return gate_result::forbidden;
    case 429:
        return gate_result::rate_limit;
    case 502:
        return gate_result::push_service_fail;
    case 503:
        return gate_result::service_unavailable;
    case 504:
        return gate_result::timeout;
    default:
        return gate_result::fail;
    }
}

inline string extract_new_token(const json_value_ref& data)
{
    return json_get(data, "new_token", string{});
}

inline bool protect_new_token(json_value_ref& data)
{
    if (auto new_token = extract_new_token(data); new_token.size())
    {
        auto len = new_token.size();
        data["new_token"] = "..." + new_token.substr(len > 12 ? len - 10 : std::min(len, 3UL));
        return true;
    }
    return false;
}

}}
