#include "http_gate.h"
#include <yxiva/core/callbacks.h>
#include <yplatform/encoding/url_encode.h>
#include <set>

namespace yxiva { namespace hub {
namespace {
static const std::set<string> REMOVABLE_SCHEMES{ callback_uri::SCHEME_XIVA_WEBSOCKET };
string remove_scheme(const string& url)
{
    auto pos = url.find(':') + 1; // 0 in case of npos, past : otherwise.
    if (!REMOVABLE_SCHEMES.count(url.substr(0, pos)))
    {
        pos = 0;
    }
    return url.substr(pos);
}
}

namespace p = std::placeholders;

http_gate::http_gate(yplatform::reactor& reactor, const http_gate_settings& settings)
    : settings_(settings), http_client_(reactor, settings.http)
{
    http_options_.timeouts.total = settings_.timeout;
}

void http_gate::send_message(const packet& packet, const handler_type& handler)
{
    try
    {
        auto callback_url = prepare_callback(packet.subscription);
        string headers = "Content-Type: application/x-msgpack\r\n";
        http_client_.async_run(
            packet.ctx,
            yhttp::request::POST(callback_url, headers, pack(packet)),
            http_options_,
            std::bind(&http_gate::handle_http_call, this, p::_1, p::_2, handler));
    }
    catch (const std::exception& e)
    {
        handler({ result::fail, e.what(), {} });
        return;
    }
}

const time_duration& http_gate::send_timeout_for(const sub_t&) const
{
    return http_options_.timeouts.total;
}

string http_gate::prepare_callback(const sub_t& subscription)
{
    string encoded_subscription_id;
    encoded_subscription_id += yplatform::url_encode(subscription.id);
    dictionary_templater::map_t values = {
        { yxiva::parameters::SUBSCRIPTION_ID, encoded_subscription_id },
        { yxiva::parameters::SUBSCRIPTION_ID_PLACEHOLDER, encoded_subscription_id }
    };
    return templater_.format(remove_scheme(subscription.callback_url), values);
}

void http_gate::handle_http_call(
    const boost::system::error_code& ec,
    yhttp::response response,
    const handler_type& handler)
{
    if (ec)
    {
        // Easy way to separate internal problems from deactivated or dead remote hosts issues.
        bool probably_host_dead =
            ec == yhttp::errc::connect_error || ec == yhttp::errc::connection_timeout;
        auto result = probably_host_dead ? result::push_service_fail : result::fail;
        handler({ result, ec.message(), {} });
    }
    else
    {
        if (response.body.size() > settings_.max_body_len)
        {
            response.body.resize(settings_.max_body_len);
        }
        auto res = gate_result_from_http_code(response.status);
        handler(
            { res,
              res == result::fail ? "http_code " + std::to_string(response.status) : response.body,
              {} });
    }
}

}}
