#include "mobile_gate.h"

#include "hacks.h"
#include <yxiva/core/operation_result.h>
#include <yxiva/core/json.h>
#include <yxiva/core/callbacks.h>
#include <yxiva/core/platforms.h>
#include <ymod_httpclient/call.h>
#include <yplatform/encoding/url_encode.h>
#include <yplatform/util/sstream.h>
#include <boost/algorithm/string/join.hpp>

namespace yxiva { namespace hub {

namespace {
static const string NO_ERROR;
}

namespace p = std::placeholders;

mobile_gate::mobile_gate(yplatform::reactor& reactor, const mobile_gate_settings& st)
    : settings_(st), http_client_(reactor, settings_.http)
{
    http_options_.reuse_connection = true;
    http_options_.timeouts.total = settings_.timeout;
    http_options_windows_.reuse_connection = true;
    http_options_windows_.timeouts.total = settings_.windows_timeout;
}

void mobile_gate::send_message(
    const packet& packet,
    push_request_cache* cache,
    const handler_type& handler)
{
    auto task = std::make_shared<context>(packet.ctx, packet.subscription);
    if (task->subscription.app_name.empty())
    {
        handler({ result::fail, "failed to retrieve app_name from subscription", {} });
        return;
    }
    if (android_platform(task->subscription.platform) && packet.ttl > settings_.fcm_max_ttl)
    {
        handler({ result::bad_request,
                  "ttl is too big",
                  { { "app_name", task->subscription.app_name } } });
        return;
    }

    task->subscription.repack_features = repack_features(packet, task->subscription, settings_);

    if (packet.message.type == message_content_type::binary)
    {
        handler({ result::ignored, "inacceptable content type", std::move(task->fields) });
        return;
    }

    if (auto repack_result = cache ?
            repack_message_if_needed(packet, task->subscription, task->messages, *cache) :
            repack_message_if_needed(packet, task->subscription, task->messages))
    {

        task->fields["bright_push"] = task->messages.is_bright() ? "true" : "false";
        task->fields["repacked_size"] = std::to_string(task->messages.payload_size());

        if (task->messages.empty())
        {
            handler({ result::ignored, "no notifications generated", std::move(task->fields) });
            return;
        }
        if (hacks::need_dump_request(packet))
        {
            hacks::dump_request(task->messages, packet.message.transit_id);
        }
        prepare_request_url(task, packet);
        do_send(task, handler);
    }
    else
    {
        handler({ result::unprocessable,
                  repack_result.error_reason,
                  { { "app_name", task->subscription.app_name } } });
    }
}

const time_duration& mobile_gate::send_timeout_for(const sub_t& subscription) const
{
    return (subscription.platform == platform::WNS || subscription.platform == platform::MPNS) ?
        http_options_windows_.timeouts.total :
        http_options_.timeouts.total;
}

void mobile_gate::do_send(std::shared_ptr<context> task, const handler_type& handler)
{
    auto& messages = task->messages;
    auto& push_msg = messages.pop_front();

    // Replace empty payload by "{}" because xivamob does not accept empty
    // strings as payload. Repacker is not responsible for that because
    // it was developed not only for mobile push notifications.
    string body = yhttp::form_encode(
                      { { "token", task->subscription.push_token },
                        { "payload", push_msg.payload().size() ? push_msg.payload() : "{}" } }) +
        push_msg.http_url_params();

    try
    {
        http_client_.async_run(
            task->ctx,
            yhttp::request::POST(task->url, std::move(body)),
            http_options_,
            std::bind(&mobile_gate::handle_send, this, p::_1, p::_2, task, handler));
    }
    catch (const std::exception& e)
    {
        handler({ result::fail, e.what(), std::move(task->fields) });
        return;
    }
}

void mobile_gate::handle_send(
    const boost::system::error_code& ec,
    yhttp::response response,
    std::shared_ptr<context> task,
    const handler_type& handler)
{
    auto& messages = task->messages;

    if (ec)
    {
        handler({ result::fail, ec.message(), std::move(task->fields) });
    }
    else
    {
        json_value resp;
        if (auto res = json_parse(resp, response.body); res)
        {
            if (auto new_token = extract_new_token(resp); !task->fields.count("new_callback") &&
                is_usable_token(new_token, task->subscription.platform))
            {
                task->fields["new_callback"] =
                    callback_uri::mobile_uri(task->subscription.app_name, new_token);
            }
            // Replace answer with edited json.
            if (protect_new_token(resp))
            {
                response.body = json_write(resp);
            }
        }
        auto res = gate_result_from_http_code(response.status);
        if (res == result::success && !messages.empty())
        {
            do_send(task, handler);
        }
        else
        {
            // Don't unsubscribe if new token is available,
            // update subscription and retry instead.
            if (res == result::unsubscribe && task->fields.count("new_callback"))
            {
                res = result::fail;
            }
            handler({ res, response.body, std::move(task->fields) });
        }
    }
}

void mobile_gate::prepare_request_url(const std::shared_ptr<context>& task, const packet& packet)
    const
{
    // APNS_QUEUE is a virtual platform for repack and subscribe, push to apns instead.
    const string& platform = platform::get_real(task->subscription.platform);

    task->url = "/push/" + platform +
        yhttp::url_encode({ { "app", task->subscription.app_name },
                            { "uid", packet.uid },
                            { "service", packet.service },
                            { "session", packet.subscription.session_key },
                            { "transit_id", packet.message.transit_id },
                            { "ttl", packet.ttl } });
}

}}
