#include "send_processor.h"

#include "convey/exhaust.h"
#include <yplatform/encoding/url_encode.h>
#include <algorithm>

#define FCM_BATCH_LOG(level)                                                                       \
    YLOG_CTX_GLOBAL(data->ctx, level) << "fcm batch processor: app_name=" << data->app_name << ", "

namespace yxiva { namespace hub {

void batch_processor::on_data(stream_ptr stream, size_t begin_id, size_t end_id)
{
    for (auto i = begin_id; i != end_id; ++i)
    {
        auto& data = stream->at(i);
        auto& initial_ctx = data->ctx->ctx;
        if (initial_ctx->is_cancelled())
        {
            input()->commit(i);
        }
        else
        {
            auto res = send_message(data, i);
            if (!res)
            {
                input()->commit(i);
                data->cb(error_code_from_gate_result(gate_result::fail), res.error_reason);
            }
        }
    }
}

operation::result batch_processor::send_message(const data_ptr& data, size_t i)
{
    try
    {
        // renew ttl
        data->pkt.ttl = remaining_ttl(*data->ctx->message);
        send_(
            data->pkt,
            false,
            [cb = data->cb, self = shared_from_this(), this, i](
                const error_code& code, const string& data) {
                input()->commit(i);
                cb(code, data);
            },
            data->ctx->cache.get());
    }
    catch (const std::exception& ex)
    {
        return ex.what();
    }
    return operation::success;
}

void fcm_batch_processor::on_data(stream_ptr stream, size_t begin_id, size_t end_id)
{
    for (auto i = begin_id; i < end_id; ++i)
    {
        auto& data = stream->at(i);
        auto& ctx = data->ctx;
        auto& initial_ctx = ctx->ctx;
        if (initial_ctx->is_cancelled() || data->subscriptions.empty())
        {
            input()->commit(i);
        }
        else if (data->req.empty())
        {
            input()->commit(i);
            report_all(data, gate_result::ignored, "no notifications generated");
        }
        else
        {
            auto res = send_batch(data, i);
            if (!res)
            {
                input()->commit(i);
                report_all(data, gate_result::fail, res.error_reason);
            }
        }
    }
}

operation::result fcm_batch_processor::send_batch(const fcm_batch_data_ptr& data, size_t i)
{
    namespace p = std::placeholders;
    using yplatform::sstream;
    using yplatform::url_encode;
    auto& ctx = data->ctx;
    auto& initial_ctx = ctx->ctx;

    try
    {
        // fcm always has no more than one message
        auto& push_msg = data->req.pop_front();

        string url = "/batch_push/fcm" +
            yhttp::url_encode({
                { "app", data->app_name },
                { "transit_id", ctx->message->transit_id },
                { "ttl", std::min(remaining_ttl(*ctx->message), settings_.fcm_max_ttl) },
            });

        // Replace empty payload by "{}" because xivamob does not accept empty
        // strings as payload. Repacker is not responsible for that because
        // it was developed not only for mobile push notifications.
        string body = yhttp::form_encode(
                          { { "payload", push_msg.payload().size() ? push_msg.payload() : "{}" },
                            { "tokens", data->tokens } }) +
            push_msg.http_url_params();

        http_client_.async_run(
            initial_ctx,
            yhttp::request::POST(url, std::move(body)),
            http_options_,
            std::bind(
                &fcm_batch_processor::handle_send,
                yplatform::shared_from(this),
                p::_1,
                p::_2,
                data,
                i));
    }
    catch (const std::exception& e)
    {
        return e.what();
    }
    return operation::success;
}

void fcm_batch_processor::handle_send(
    const boost::system::error_code& ec,
    yhttp::response response,
    fcm_batch_data_ptr data,
    size_t i)
{
    if (ec)
    {
        input()->commit(i);
        FCM_BATCH_LOG(error) << ec.message();
        report_all(data, gate_result::fail, ec.message());
    }
    else if (response.status != 200)
    {
        input()->commit(i);
        FCM_BATCH_LOG(error) << "response code " << response.status << ", body " << response.body;
        report_all(data, gate_result_from_http_code(response.status), response.body);
    }
    else
    {
        auto res = handle_string_response(response.body, data);
        input()->commit(i);
        if (!res)
        {
            FCM_BATCH_LOG(error) << res.error_reason << ", body " << response.body;
            report_all(data, gate_result::fail, res.error_reason);
        }
    }
}

operation::result fcm_batch_processor::handle_string_response(
    const string& body,
    const fcm_batch_data_ptr& data)
{
    using yplatform::sstream;
    static const char key_responses[] = "results";

    try
    {
        json_value json_resp;
        auto res = json_parse(json_resp, body);
        if (!res || !json_resp.has_member(key_responses) || !json_resp[key_responses].is_array())
        {
            string resp;
            sstream(resp, res.error_reason.size() + 48)
                << "malformed response: " << (res ? "invalid structure" : res.error_reason);
            return resp;
        }

        auto&& responses = json_resp[key_responses];
        if (responses.size() != data->subscriptions.size())
        {
            string resp;
            sstream(resp, 72) << "malformed response: "
                              << "expected " << data->subscriptions.size() << " responses, but "
                              << responses.size() << " received";
            return resp;
        }

        FCM_BATCH_LOG(info) << "batch of " << data->subscriptions.size()
                            << " subscriptions sent successfully";
        handle_json_responses(responses, data);
    }
    catch (const std::exception& ex)
    {
        return ex.what();
    }
    return operation::success;
}

void fcm_batch_processor::handle_json_responses(
    json_value_ref& responses,
    const fcm_batch_data_ptr& data)
{
    static const char key_result[] = "result";
    static const char key_code[] = "code";

    for (size_t j = 0; j < data->subscriptions.size(); ++j)
    {
        try
        {
            auto&& resp = responses[static_cast<size_t>(j)];
            // log if response does not contain code
            if (!resp.has_member(key_code) ||
                (resp[key_code].is_int64() && resp[key_code].is_uint64()))
            {
                FCM_BATCH_LOG(info) << "sub_id " << data->subscriptions[j]->id
                                    << ", malformed response: \"" << json_write(resp) << "\"";
            }
            auto code = json_get<unsigned>(resp, key_code, 0);
            string result;
            if (resp.has_member(key_result))
            {
                auto&& r = resp[key_result];
                if (auto new_token = extract_new_token(r);
                    is_usable_token(new_token, platform::FCM))
                {
                    data->fields["new_callback"] =
                        callback_uri::mobile_uri(data->app_name, new_token);
                }
                protect_new_token(r);
                result = (r.is_object() || r.is_array()) ? json_write(r) : json_get<string>(r, "");
            }
            data->cb(data, j, { gate_result_from_http_code(code), result, data->fields });
        }
        catch (const std::exception& ex)
        {
            data->cb(data, j, { gate_result::fail, ex.what(), data->fields });
        }
    }
}

}}
