#pragma once

#include "batch_data.h"
#include "settings.h"
#include <pipeline/processor.h>
#include <pipeline/stream_strand.h>
#include <yxiva/core/json.h>
#include <ymod_httpclient/cluster_client.h>

namespace yxiva { namespace hub {

static constexpr size_t FCM_MAX_BATCH_SIZE = 1000;

template <typename DataType>
using processor = pipeline::Processor<pipeline::StreamStrand<DataType>>;
using pipe_settings = pipeline::StreamSettings;

class batch_processor : public processor<data_ptr>
{
    using base_t = processor<data_ptr>;
    // exhaust::send
    using exhaust_send_t =
        std::function<void(const packet&, bool, const convey_callback_t&, push_request_cache*)>;

public:
    batch_processor(
        boost::asio::io_service& io,
        const pipe_settings& st,
        const exhaust_send_t& send)
        : base_t(io, st), send_(send)
    {
        input()->label("batch_proc");
    }

private:
    void on_data(stream_ptr stream, size_t begin_id, size_t end_id) override;
    operation::result send_message(const data_ptr& data, size_t i);

    exhaust_send_t send_;
};

class fcm_batch_processor : public processor<fcm_batch_data_ptr>
{
    using base_t = processor<fcm_batch_data_ptr>;
    using http_client_ptr = shared_ptr<yhttp::call>;

public:
    fcm_batch_processor(
        yplatform::reactor& reactor,
        const pipe_settings& st,
        const mobile_gate_settings& settings)
        : base_t(*reactor.io(), st), http_client_(reactor, settings.http), settings_(settings)
    {
        input()->label("fcm_batch_proc");
        http_options_.reuse_connection = true;
        http_options_.timeouts.total = settings_.timeout;
    }

private:
    void on_data(stream_ptr stream, size_t begin_id, size_t end_id) override;
    operation::result send_batch(const fcm_batch_data_ptr& data, size_t i);
    void handle_send(
        const boost::system::error_code& ec,
        yhttp::response response,
        fcm_batch_data_ptr data,
        size_t i);
    operation::result handle_string_response(const string& body, const fcm_batch_data_ptr& data);
    void handle_json_responses(json_value_ref& responses, const fcm_batch_data_ptr& data);

    void report_all(const fcm_batch_data_ptr& data, gate_result res, const string& body)
    {
        for (size_t i = 0; i < data->subscriptions.size(); ++i)
        {
            data->cb(data, i, gate_response{ res, body, {} });
        }
    }

    yhttp::cluster_client http_client_;
    const mobile_gate_settings settings_;
    yhttp::cluster_client::options http_options_;
};

}}
