#pragma once

#include <yxiva/core/types.h>
#include <yxiva/core/gid.h>
#include <yxiva/core/message.h>
#include <yxiva/core/platforms.h>
#include <yxiva/core/feature_toggling.h>
#include <yxiva/core/repacker.h>
#include <ymod_httpclient/client.h>
#include <ymod_httpclient/cluster_client.h>

namespace yxiva { namespace hub {
namespace {
inline auto load_string_set(const yplatform::ptree& conf, const string& name)
{
    std::set<string> string_set;
    auto range = conf.equal_range(name);
    for (auto it = range.first; it != range.second; ++it)
    {
        string_set.insert(it->second.data());
    }
    return string_set;
}
}

// TODO simplify transit_id_injection and collapse_id_injection structs
struct experiments_settings
{
    struct
    {
        // Stop list for injection.
        std::set<string> app_blacklist;
        // Force injections for apps in stop list if uid in whitelist.
        std::map<string, std::set<string>> uids_whitelist;
        // Force injections for uids in experiment.
        // The second element in a pair is a percent value scaled to 1000.
        std::map<string, unsigned> experiment;

        // Returns true if transit id should be injected to payload.
        bool match(const string& full_app_name, const string& uid) const
        {
            auto&& key = full_app_name;
            if (app_blacklist.count(key) == 0) return true;
            if (uids_whitelist.count(key) && uids_whitelist.at(key).count(uid)) return true;
            auto gid = gid_from_uid(uid);
            if (experiment.count(key) && gid % 1000 < experiment.at(key)) return true;
            return false;
        }

        void load(const yplatform::ptree& conf)
        {
            auto range = conf.equal_range("app_blacklist");
            for (auto it = range.first; it != range.second; ++it)
            {
                app_blacklist.insert(it->second.data());
            }

            range = conf.equal_range("uids_whitelist");
            for (auto it = range.first; it != range.second; ++it)
            {
                auto& node = it->second;
                auto& list = uids_whitelist[node.get<string>("app")];
                auto uids_range = node.equal_range("uids");
                for (auto jt = uids_range.first; jt != uids_range.second; ++jt)
                {
                    list.insert(jt->second.data());
                }
            }
        }
    } transit_id_injection;

    struct
    {
        std::set<string> service_whitelist;

        bool match(const sub_t& subscription) const
        {
            return service_whitelist.count(subscription.service) > 0;
        }

        void load(const yplatform::ptree& conf)
        {
            service_whitelist = load_string_set(conf, "service_whitelist");
        }

    } collapse_id_injection;

    whitelisted_feature transit_id_injection_appmetrica_by_app;

    void load(const yplatform::ptree& conf)
    {
        transit_id_injection.load(conf.get_child("transit_id_injection"));
        collapse_id_injection.load(conf.get_child("collapse_id_injection"));
        transit_id_injection_appmetrica_by_app.load(
            conf.get_child("transit_id_injection_appmetrica"), "app_whitelist");
    }
};

struct http_gate_settings
{
    time_duration timeout;
    size_t max_body_len = 1000;
    yhttp::client::settings http;

    void load(const yplatform::ptree& conf)
    {
        timeout = conf.get<time_duration>("send_timeouts.http");
        max_body_len = conf.get("max_gate_reply_body_size", max_body_len);

        http.parse_ptree(conf.get_child("http"));
    }
};

struct mobile_gate_settings
{
    time_duration timeout;
    time_duration windows_timeout;
    ttl_t fcm_max_ttl;
    experiments_settings experiments;
    yhttp::cluster_client::settings http;

    void load(const yplatform::ptree& conf)
    {
        timeout = conf.get<time_duration>("send_timeouts.mobile");
        windows_timeout = conf.get<time_duration>("send_timeouts.mobile_windows");
        fcm_max_ttl = conf.get<ttl_t>("fcm_max_ttl");

        experiments.load(conf.get_child("experiments"));
        http.parse_ptree(conf.get_child("http"));
        http.nodes = { conf.get<string>("mobile_gate") };
        http.preffered_pool_size = conf.get("xivamob_pool_size", http.preffered_pool_size);
    }
};

struct webpush_gate_settings
{
    time_duration timeout;
    yhttp::cluster_client::settings http;
    std::set<string> services_with_webpush_transit_id_injection;
    std::set<string> services_with_webpush_uts_injection;

    void load(const yplatform::ptree& conf)
    {
        timeout = conf.get<time_duration>("send_timeouts.webpush");
        http.parse_ptree(conf.get_child("http"));
        http.nodes = { conf.get<string>("mobile_gate") };
        http.preffered_pool_size = conf.get("xivamob_pool_size", http.preffered_pool_size);
        services_with_webpush_transit_id_injection =
            load_string_set(conf, "services_with_webpush_transit_id_injection");
        services_with_webpush_uts_injection =
            load_string_set(conf, "services_with_webpush_uts_injection");
    }
};

struct apns_queue_gate_settings
{
    size_t max_payload_size;
    bool suppress_apns_queue_repack;
    yhttp::cluster_client::options http_options;
    yhttp::cluster_client::settings http;

    void load(const yplatform::ptree& conf)
    {
        max_payload_size = conf.get<size_t>("apns_queue_gate.max_payload_size");
        suppress_apns_queue_repack = conf.get("suppress_apns_queue_repack", false);

        http.parse_ptree(conf.get_child("http"));
        http.nodes = { conf.get<string>("apns_queue_gate.hub_url") };

        http_options.reuse_connection = true;
        http_options.timeouts.total = conf.get<time_duration>("send_timeouts.http");
    }
};

struct websocket_gate_settings : http_gate_settings
{
    void load(const yplatform::ptree& conf)
    {
        http_gate_settings::load(conf);
        http.ssl_verify_hostname = conf.get("websocket_verify_hostname", http.ssl_verify_hostname);
        http.user_agent = conf.get("websocket_user_agent", http.user_agent);
    }
};

struct exhaust_settings
{
    unsigned drop_overlap_sec = 5;

    http_gate_settings http;
    websocket_gate_settings ws;
    mobile_gate_settings mobile;
    webpush_gate_settings webpush;
    apns_queue_gate_settings apns_queue;

    void load(yplatform::ptree const& conf)
    {
        drop_overlap_sec = time_traits::get_seconds_count(
            conf.get<time_duration>("drop_overlap", seconds(drop_overlap_sec)));

        http.load(conf);
        ws.load(conf);
        mobile.load(conf);
        webpush.load(conf);
        apns_queue.load(conf);
    }
};

inline auto repack_features(
    const packet& packet,
    const push_subscription_params& mob_sub,
    const mobile_gate_settings& settings)
{
    auto full_app_name = platform::resolve_alias(mob_sub.platform).name + ":" + mob_sub.app_name;
    struct repack_features result;
    result.inject_transit_id =
        settings.experiments.transit_id_injection.match(full_app_name, packet.message.uid);
    if (result.inject_transit_id)
    {
        result.transit_id_appmetrica_format =
            settings.experiments.transit_id_injection_appmetrica_by_app.enabled_for(full_app_name);
    }
    result.inject_collapse_id = settings.experiments.collapse_id_injection.match(mob_sub);
    return result;
}

inline auto repack_features(const packet& packet, const webpush_gate_settings& settings)
{
    struct repack_features result;
    if (settings.services_with_webpush_transit_id_injection.count(packet.service))
    {
        result.inject_transit_id = true;
    }
    if (settings.services_with_webpush_uts_injection.count(packet.service))
    {
        result.inject_uts = true;
    }
    return result;
}

}}