#include "webpush_gate.h"

#include <yxiva/core/binary_protocol.h>
#include <yxiva/core/operation_result.h>
#include <yxiva/core/json.h>
#include <yxiva/core/callbacks.h>
#include <yxiva/core/platforms.h>
#include <yxiva/core/repacker.h>
#include <ymod_httpclient/call.h>
#include <yplatform/encoding/url_encode.h>

namespace yxiva { namespace hub {

namespace p = std::placeholders;

webpush_gate::webpush_gate(yplatform::reactor& reactor, const webpush_gate_settings& st)
    : settings_(st), http_client_(reactor, settings_.http)
{
    http_options_.timeouts.total = settings_.timeout;
}

void webpush_gate::send_message(const packet& packet, const handler_type& handler)
{
    try
    {
        string webpush_subscription;
        auto& uri = packet.subscription.callback_url;
        auto parsed = callback_uri::parse_webpush_uri(uri, webpush_subscription);
        if (!parsed)
        {
            handler({ result_type::fail, parsed.error_reason, {} });
            return;
        }

        auto request = yhttp::request::POST(
            prepare_request_url(packet), prepare_request_body(packet, webpush_subscription));
        http_client_.async_run(
            packet.ctx,
            std::move(request),
            std::bind(&webpush_gate::handle_send, this, p::_1, p::_2, handler));
    }
    catch (const std::exception& e)
    {
        handler({ result_type::fail, e.what(), {} });
    }
}

const time_duration& webpush_gate::send_timeout_for(const sub_t&) const
{
    return http_options_.timeouts.total;
}

void webpush_gate::handle_send(
    const boost::system::error_code& ec,
    yhttp::response response,
    const handler_type& handler)
{
    if (ec)
    {
        handler({ result_type::fail, ec.message(), {} });
    }
    else
    {
        handler({ gate_result_from_http_code(response.status), response.body, {} });
    }
}

string webpush_gate::prepare_request_url(const packet& packet)
{
    return "/webpush" +
        yhttp::url_encode({ { "uid", packet.uid },
                            { "service", packet.service },
                            { "session", packet.subscription.session_key },
                            { "transit_id", packet.message.transit_id },
                            { "ttl", packet.ttl } });
}

string webpush_gate::prepare_request_body(
    const packet& packet,
    const string& urlencoded_webpush_subscription)
{
    string hdr; // empty for binaries
    if (packet.message.type == message_content_type::binary)
    {
        hdr = binary_protocol::pack(binary_protocol::notification_header{
            packet.uid, packet.service, packet.message.operation, packet.message.transit_id });
    }

    string req_body;
    yplatform::sstream sstream(
        req_body,
        hdr.size() + packet.message.raw_data.size() + urlencoded_webpush_subscription.size() + 64);
    sstream << "subscription=" << urlencoded_webpush_subscription
            << "&payload=" << yplatform::url_encode(hdr);
    append_request_body_payload(sstream, packet);
    return req_body;
}

void webpush_gate::append_request_body_payload(yplatform::sstream& sstream, const packet& packet)
{
    webpush_repacker repacker;
    sstream << yplatform::url_encode(repacker.repack(packet, repack_features(packet, settings_)));
}

}}
