#pragma once

#include "convey/gate.h"
#include <boost/system/error_code.hpp>
#include <ymod_webserver/codes.h>
#include <vector>

namespace yxiva { namespace hub {

using boost::system::error_code;

enum err_code : int
{
    err_code_no_error = 0,
    err_code_unknown_error = 1,
    err_code_auth_fail = 2,
    err_code_storage_fail = 3,
    err_code_agent_fail = 4,
    err_code_cancelled = 5,
    err_code_requests_collision = 6,
    err_code_no_such_service = 7,
    err_code_service_disabled = 8,
    err_code_position_update_fail = 9,
    err_code_position_update_conflict = 10,
    err_code_unsubscribe_fail = 11,
    err_code_gate_fail = 12,
    err_code_subscription_dropped = 13,
    err_code_filtered = 14,
    err_code_no_subscriptions = 15,
    err_code_agent_bad_request = 16,
    err_code_rate_limit = 17,
    err_code_push_service_fail = 18,
    err_code_agent_timeout = 19,
    err_code_service_unavailable = 20,
    err_code_forbidden = 21,
    err_code_subscription_deactivated = 22,
    err_code_invalid_uidset_uid_shard = 23,
    err_code_resharding_invalid_callback_params = 24,
    err_code_remote_xtable_fail = 25,
    err_code_migration_storage_fail = 26,
    err_code_resharding_role_mismatch = 27,
    err_code_resharding_queue_already_exists = 28,
    err_code_resharding_queue_not_found = 29,
    err_code_resharding_queue_overflow = 30,
    err_code_resharding_queue_reject = 31,
    err_code_resharding_bad_migration_state = 32,
    err_code_resharding_disabled = 33,
    err_code_unprocessable = 34,
};

// See boost::system description
class error_category : public boost::system::error_category
{
public:
    const char* name() const noexcept(true)
    {
        return "xivahub";
    }

    string message(int ev) const
    {
        return error_names.at(ev);
    }

private:
    static const std::vector<string> error_names;
};

const error_category& category();

inline error_code make_error_code(err_code code = err_code_no_error)
{
    return error_code(code, category());
}

inline error_code error_code_from_gate_result(gate_result res)
{
    switch (res)
    {
    case gate_result::success:
    case gate_result::ignored:
        return make_error_code();
    case gate_result::deactivate:
        return make_error_code(err_code_subscription_deactivated);
    case gate_result::unsubscribe:
        return make_error_code(err_code_subscription_dropped);
    case gate_result::fail:
        return make_error_code(err_code_gate_fail);
    case gate_result::bad_request:
        return make_error_code(err_code_agent_bad_request);
    case gate_result::rate_limit:
        return make_error_code(err_code_rate_limit);
    case gate_result::push_service_fail:
        return make_error_code(err_code_push_service_fail);
    case gate_result::timeout:
        return make_error_code(err_code_agent_timeout);
    case gate_result::service_unavailable:
        return make_error_code(err_code_service_unavailable);
    case gate_result::forbidden:
        return make_error_code(err_code_forbidden);
    case gate_result::unprocessable:
        return make_error_code(err_code_unprocessable);
    default:
        return make_error_code(err_code_unknown_error);
    }
}

inline ymod_webserver::codes::code http_code_for_error(const error_code& e)
{
    using ymod_webserver::codes::code;
    if (e && e.category() != category()) return code::internal_server_error;

    switch (e.value())
    {
    case err_code_no_error:
        return code::ok;
    case err_code_filtered:
        return code::accepted;
    case err_code_no_subscriptions:
        return code::no_content;
    case err_code_subscription_dropped:
        return code::reset_content;
    case err_code_agent_bad_request:
        return code::bad_request;
    case err_code_forbidden:
        return code::forbidden;
    case err_code_unprocessable:
        return code::unprocessable;
    case err_code_rate_limit:
        return code::too_many_requests;
    case err_code_push_service_fail:
        return code::bad_gateway;
    case err_code_service_unavailable:
        return code::service_unavailable;
    case err_code_agent_timeout:
        return code::gateway_timeout;
    case err_code_service_disabled:
        return code::forbidden;
    case err_code_no_such_service:
        return code::bad_request;
    case err_code_invalid_uidset_uid_shard:
        return code::bad_request;
    case err_code_resharding_role_mismatch:
        return code::conflict;
    case err_code_resharding_bad_migration_state:
        return code::conflict;
    default:
        return code::internal_server_error;
    }
}

inline string message_for_error(const error_code& err)
{
    return err ? err.message() : string();
}

}}
