#include "main_module.h"

#include "xtable/xtable.h"
#include "api/bind.h"

#include <sintimers/queue.h>
#include <ymod_webserver/server.h>
#include <yplatform/find.h>
#include <yplatform/task_context.h>
#include <yplatform/util/split.h>

namespace yxiva { namespace hub {

main_module::main_module(yplatform::reactor& reactor, const yplatform::ptree& xml)
{
    unsigned self_diag_samples = xml.get<unsigned>("self_diag_samples");
    if (!self_diag_samples)
    {
        throw std::domain_error("invalid self_diag_samples value");
    }
    state_->stats.self_diag.init(self_diag_samples);

    state_->settings = make_shared<settings_t>();
    state_->settings->read(xml);
    if (state_->settings->ignore_xstore_errors)
    {
        state_->stats.ignore_xstore_errors = true;
        YLOG_L(warning) << "ignore_xstore_errors ON";
    }
    if (state_->settings->force_weak_delivery)
    {
        YLOG_L(warning) << "force_weak_delivery ON";
    }
    state_->stats.deduplicate_xtable = state_->settings->deduplicate_xtable;
    if (!state_->stats.deduplicate_xtable)
    {
        YLOG_L(warning) << "deduplicate_xtable OFF";
    }

    state_->netch = yplatform::find<ymod_messenger::module, std::shared_ptr>("netch");
    state_->sync_netch = yplatform::find<ymod_messenger::module, std::shared_ptr>("detached_netch");
    state_->lease_netch = yplatform::find<ymod_messenger::module, std::shared_ptr>("lease_netch");

    string lease_node_name = xml.get("lease_node", "lease_node");
    state_->lease_node = yplatform::find<ymod_lease::node>(lease_node_name);

    state_->paxos_node = yplatform::find<ymod_paxos::node>(xml.get("paxos_node", "paxos_node"));

    string xtasks_name = xml.get("xtasks_module", "xtasks");
    state_->xtasks = yplatform::find<ymod_xtasks::xtasks>(xtasks_name);

    timers_.reset(new timers::queue(yplatform::reactor::make_not_owning_copy(reactor)));

    xtasks_service::settings_ptr xtasks_settings(new xtasks_service::settings());
    xtasks_settings->xtasks = state_->xtasks;
    xtasks_settings->parse_ptree(xml.get_child("xtasks_service_job"));
    state_->xtasks_service.reset(new xtasks_service(timers_, xtasks_settings));

    state_->xtasks_service->logger(this->logger());

    state_->xtasks_worker.reset(new xtasks_worker());
    state_->xtasks_worker->logger(this->logger());
    state_->xtasks_worker->logger().append_log_prefix("xtasks_worker");
    state_->xtasks_worker->init(xml.get_child("xtasks_worker"));

    state_->transport_log = yplatform::find<mod_log>("xivahub_log");

    auto mod_httpclient = yplatform::find<yhttp::call>("http_client");
    state_->xtable = yplatform::find<XTable>("xtable");
    state_->xstore = yplatform::find<ymod_xstore::xstore>("xstore");
    state_->exhaust = yplatform::find<::yxiva::hub::exhaust>("exhaust");

    state_->pq_xtable = yplatform::find<ymod_pq::cluster_call>("pq_xtable");
    state_->pq_xstore = yplatform::find<ymod_pq::cluster_call>("pq_xstore");
    state_->pq_batch = yplatform::find<ymod_pq::cluster_call>("pq_batch");

    if (state_->settings->cluster.use_static_metadata)
    {
        state_->leasemeta = state_->lease_node;
    }
    else
    {
        local_reactor_ = std::make_shared<yplatform::reactor>();
        local_reactor_->init(1, 1);
        auto messenger =
            std::make_shared<ymod_messenger::module>(*local_reactor_, yplatform::ptree{});
        auto messenger_logger = logger();
        messenger_logger.set_log_prefix("leasemeta_netch");
        messenger->logger(messenger_logger);
        if (state_->settings->cluster.leasemeta.arbiters.empty())
        {
            throw std::domain_error("empty leasemeta arbiters list");
        }
        messenger->connect_to_cluster(state_->settings->cluster.leasemeta.arbiters);
        yplatform::register_module(*local_reactor_, "netch", messenger);
        ylease::settings lease_conf;
        lease_conf.lease_log_id = "";
        lease_conf.netch_module_name = "netch";
        lease_conf.arbiters_count = state_->settings->cluster.leasemeta.arbiters.size();
        lease_conf.acquire_lease_timeout =
            state_->settings->cluster.leasemeta.acquire_lease_timeout;
        lease_conf.max_lease_time = state_->settings->cluster.leasemeta.max_lease_time;
        lease_conf.verbose_logging = state_->settings->cluster.leasemeta.verbose_logging;
        lease_conf.tag = state_->settings->cluster.name;
        state_->leasemeta = boost::make_shared<ylease::node>(*local_reactor_, lease_conf);
        auto lease_logger = logger();
        lease_logger.set_log_prefix("leasemeta");
        state_->leasemeta->logger(lease_logger);
    }

    control_module_ = std::make_shared<control_module>(reactor, state_);
    state_->controller = control_module_; // weak

    finalize_timer_ = timers_->create_timer();

    bind_api(state_);
}

void main_module::init()
{
    // With leasemeta force control_module to complete initial
    // lease acquiring and pick cluster configuration.
    if (local_reactor_) local_reactor_->run();

    control_module_->logger(logger());
    control_module_->logger().set_log_prefix("cluster_control");
    control_module_->init();

    if (state_->settings->cluster.use_static_metadata) return;

    auto ticks = state_->settings->cluster.leasemeta.init_delay / milliseconds(10);
    for (int i = 0; i < ticks && !state_->stats.convey_enabled; ++i)
    {
        std::this_thread::sleep_for(milliseconds(10));
    }

    if (!state_->stats.convey_enabled)
    {
        if (local_reactor_) local_reactor_->fini();
        throw std::domain_error("failed to get cluster meta or convey is disabled");
    }

    if (state_->stats.finalize)
    {
        if (local_reactor_) local_reactor_->fini();
        throw std::domain_error("app finished by cluster control");
    }
}

void main_module::start()
{
    control_module_->start();
    if (!state_->settings->force_weak_delivery)
    {
        state_->xtasks_worker->start();
    }

    start_finalize_timer();
}

void main_module::start_finalize_timer(bool last_value)
{
    finalize_timer_->async_wait(state_->settings->finalize_delay, [this, last_value]() {
        if (state_->stats.finalize && last_value)
        {
            YLOG_L(notice) << "finalize";
            kill(getpid(), SIGINT);
        }
        else
        {
            start_finalize_timer(state_->stats.finalize);
        }
    });
}

void main_module::stop()
{
    control_module_->stop();
    if (!state_->settings->force_weak_delivery)
    {
        state_->xtasks_worker->stop();
    }
}

void main_module::fini()
{
    if (local_reactor_)
    {
        local_reactor_->stop();
        local_reactor_->fini();
    }
}

yplatform::ptree main_module::get_stats() const
{
    yplatform::ptree result;
    state_->stats.to_ptree().swap(result);
    state_->xtasks_worker->get_stats().swap(result.add_child("worker", yplatform::ptree()));
    return result;
}

}}
