#pragma once

#include <yxiva/core/callbacks.h>
#include <yxiva/core/types.h>
#include <yxiva/core/message.h>
#include <yxiva/core/subscriptions.h>
#include <yxiva/core/map_utils.h>
#include <ymod_xstore/types.h>
#include <yplatform/module.h>
#include <yplatform/log/typed.h>
#include <yplatform/task_context.h>
#include <yplatform/util/sstream.h>
#include <yplatform/hash/md5.h>
#include <boost/algorithm/string/join.hpp>
#include <boost/lexical_cast.hpp>

#define N_COMMON YLOG((*logger_), info) << root_am_notifications
#define N_SPECIAL YLOG((*logger_), info) << root_am_notifications_special
#define S_COMMON YLOG((*logger_), info) << root_am_subscriptions
#define S_SPECIAL YLOG((*logger_), info) << root_am_subscriptions_special

#define make_context_attr(ctx) ll::make_attr("context", ctx ? ctx->uniq_id() : "")

namespace yxiva { namespace hub {

namespace ll = yplatform::log::typed;

using callback_uri::is_mobile_uri;
using callback_uri::parse_mobile_uri;
using callback_uri::is_apns_queue_uri;
using callback_uri::parse_apns_queue_uri;
using callback_uri::is_webpush_uri;
using callback_uri::parse_webpush_uri_origin;
using callback_uri::is_xiva_websocket_uri;

class mod_log : public yplatform::module
{
public:
    void init(const yplatform::ptree& conf)
    {
        parse_notification_fields(conf.get("log_notification_fields", ""));
        auto typed_log_id = conf.get("typed_log_id", "tskv");
        logger_ = yplatform::log::tskv_logger(YGLOBAL_LOG_SERVICE, typed_log_id);
        root_am_notifications << ll::make_attr("group", "notifications");
        root_am_notifications_special << ll::make_attr("group", "notifications-special");
        root_am_subscriptions << ll::make_attr("group", "subscriptions");
        root_am_subscriptions_special << ll::make_attr("group", "subscriptions-special");
    }

    const char* bool_message(bool val)
    {
        return val ? "yes" : "no";
    }

    void new_notification(
        task_context_ptr ctx,
        const message& message,
        bool is_batch = false,
        const string& hash = {})
    {
        auto arrival_ts = is_batch ? std::time(nullptr) : message.event_ts;
        N_SPECIAL << ll::make_attr("status", "new") << make_context_attr(ctx) << make_attrs(message)
                  << ll::make_attr("hash", hash)
                  << ll::make_attr("arrival_time", std::to_string(arrival_ts))
                  << ll::make_attr("is_batch", bool_message(is_batch));
    }

    void notification_saved(
        task_context_ptr ctx,
        const message& message,
        ymod_xstore::merge_type merge,
        const string old_transit_id)
    {
        bool is_duplicate = merge == ymod_xstore::merge_type::duplicate;
        bool is_merged_by_topic = merge == ymod_xstore::merge_type::topic;
        ll::attributes_map optional_attrs;
        if (old_transit_id.size())
        {
            optional_attrs << ll::make_attr("old_transit_id", old_transit_id);
        }

        N_SPECIAL << ll::make_attr("status", "saved") << make_context_attr(ctx)
                  << make_attrs(message) << ll::make_attr("duplicate", bool_message(is_duplicate))
                  << ll::make_attr("merged_by_topic", bool_message(is_merged_by_topic))
                  << optional_attrs;
    }

    void notification_save_failed(task_context_ptr ctx, const message& message, const string& error)
    {
        N_SPECIAL << ll::make_attr("status", "save failed") << make_context_attr(ctx)
                  << make_attrs(message) << ll::make_attr("error", error);
    }

    void notification_task_created(task_context_ptr ctx, const message& message)
    {
        N_SPECIAL << ll::make_attr("status", "xtask created") << make_context_attr(ctx)
                  << make_attrs(message);
    }

    void notification_task_create_failed(
        task_context_ptr ctx,
        const message& message,
        const string& error)
    {
        N_SPECIAL << ll::make_attr("status", "xtask create failed") << make_context_attr(ctx)
                  << make_attrs(message) << ll::make_attr("error", error);
    }

    void notification_convey_disabled(task_context_ptr ctx, const message& message)
    {
        N_COMMON << ll::make_attr("status", "convey disabled") << make_context_attr(ctx)
                 << make_attrs(message);
    }

    void convey_failed(
        const packet& packet,
        const string& error,
        const std::map<string, string>& info_fields)
    {
        N_COMMON << ll::make_attr("status", "convey failed")
                 << make_field_attrs(packet, info_fields) << ll::make_attr("error", error);
    }

    void convey_deactivated(const packet& packet, const std::map<string, string>& info_fields)
    {
        N_COMMON << ll::make_attr("status", "deactivated") << make_field_attrs(packet, info_fields);
    }

    void convey_dropped(
        const packet& packet,
        const string& reason,
        const std::map<string, string>& info_fields)
    {
        N_COMMON << ll::make_attr("status", "dropped") << ll::make_attr("reason", reason)
                 << make_field_attrs(packet, info_fields);
    }

    void convey_skip(const packet& packet, const string& reason)
    {
        N_COMMON << ll::make_attr("status", "skip") << ll::make_attr("reason", reason)
                 << make_field_attrs(packet);
    }

    void convey_skip_filtered(const packet& packet)
    {
        convey_skip(packet, "filtered");
    }

    void convey_ignored(
        const packet& packet,
        const string& reason,
        const std::map<string, string>& info_fields)
    {
        N_COMMON << ll::make_attr("status", "ignored") << ll::make_attr("reason", reason)
                 << make_field_attrs(packet, info_fields);
    }

    void convey_success(
        const packet& packet,
        const string& data = string(),
        const std::map<string, string>& info_fields = {})
    {
        N_COMMON << ll::make_attr("status", "conveyed") << make_field_attrs(packet, info_fields)
                 << ll::make_attr("gate_data", data);
    }

    void convey_callback_update_failed(const packet& packet, const string& error)
    {
        S_SPECIAL << ll::make_attr("status", "post-convey callback update failed")
                  << make_field_attrs(packet) << ll::make_attr("error", error);
    }

    void convey_callback_update_success(const packet& packet)
    {
        S_COMMON << ll::make_attr("status", "post-convey callback update finished")
                 << make_field_attrs(packet);
    }

    void convey_unsubscribe_failed(const packet& packet, const string& error)
    {
        S_SPECIAL << ll::make_attr("status", "post-convey unsubscribe failed")
                  << make_field_attrs(packet) << ll::make_attr("error", error);
    }

    void convey_unsubscribe_success(const packet& packet)
    {
        S_SPECIAL << ll::make_attr("status", "post-convey unsubscribe finished")
                  << make_field_attrs(packet);
    }

    void convey_update_failed(const packet& packet, const string& error)
    {
        S_SPECIAL << ll::make_attr("status", "post-convey update failed")
                  << make_field_attrs(packet) << ll::make_attr("error", error);
    }

    void convey_update_success(const packet& packet)
    {
        S_SPECIAL << ll::make_attr("status", "post-convey update finished")
                  << make_field_attrs(packet);
    }

    void convey_job_update_failed(
        task_context_ptr ctx,
        const sub_t& subscription,
        local_id_t final_id,
        const string& error)
    {
        S_SPECIAL << ll::make_attr("status", "job update failed")
                  << ll::make_attr("final_id", final_id) << ll::make_attr("local_id", final_id)
                  << ll::make_attr("error", error) << make_field_attrs(subscription)
                  << make_context_attr(ctx);
    }

    void convey_job_update_success(
        task_context_ptr ctx,
        const sub_t& subscription,
        local_id_t final_id)
    {
        S_SPECIAL << ll::make_attr("status", "job update finished")
                  << ll::make_attr("final_id", final_id) << ll::make_attr("local_id", final_id)
                  << make_field_attrs(subscription) << make_context_attr(ctx);
    }

    void convey_expired(const packet& packet)
    {
        N_COMMON << ll::make_attr("status", "expired") << make_field_attrs(packet);
    }

    void subscribe_convey_disabled(task_context_ptr ctx, const sub_t& subscription)
    {
        S_SPECIAL << ll::make_attr("status", "convey disabled") << make_field_attrs(subscription)
                  << make_context_attr(ctx);
    }

    void subscribe_task_created(task_context_ptr ctx, const sub_t& subscription)
    {
        S_SPECIAL << ll::make_attr("status", "xtask created") << make_field_attrs(subscription)
                  << make_context_attr(ctx);
    }

    void subscribe_finished(task_context_ptr ctx, const sub_t& subscription)
    {
        S_COMMON << ll::make_attr("status", "subscribe finished") << make_field_attrs(subscription)
                 << make_context_attr(ctx);
    }

    void subscribe_failed(task_context_ptr ctx, const sub_t& subscription, const string& error)
    {
        S_COMMON << ll::make_attr("status", "subscribe failed") << make_field_attrs(subscription)
                 << ll::make_attr("error", error) << make_context_attr(ctx);
    }

    void unsubscribe_finished(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& subscription_id)
    {
        S_COMMON << ll::make_attr("status", "unsubscribe finished") << ll::make_attr("uid", uid)
                 << ll::make_attr("service", service)
                 << ll::make_attr("subscription_id", subscription_id) << make_context_attr(ctx);
    }

    void unsubscribe_failed(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& subscription_id,
        const string& error)
    {
        S_COMMON << ll::make_attr("status", "unsubscribe failed") << ll::make_attr("uid", uid)
                 << ll::make_attr("service", service)
                 << ll::make_attr("subscription_id", subscription_id)
                 << ll::make_attr("error", error) << make_context_attr(ctx);
    }

    void subscribe_task_create_failed(
        task_context_ptr ctx,
        const sub_t& subscription,
        const string& error)
    {
        S_SPECIAL << ll::make_attr("status", "xtask create failed")
                  << make_field_attrs(subscription) << ll::make_attr("error", error)
                  << make_context_attr(ctx);
    }

    void update_callback_finished(
        task_context_ptr ctx,
        const string& uidset,
        const string& service,
        const string& old_cb,
        const string& new_cb)
    {
        S_COMMON << ll::make_attr("status", "update_callback finished")
                 << ll::make_attr("uidset", uidset) << ll::make_attr("service", service)
                 << ll::make_attr("old_cb", old_cb) << ll::make_attr("new_cb", new_cb)
                 << make_context_attr(ctx);
    }

    void update_callback_failed(
        task_context_ptr ctx,
        const string& uidset,
        const string& service,
        const string& error,
        const string& old_cb,
        const string& new_cb)
    {
        S_SPECIAL << ll::make_attr("status", "update_callback failed")
                  << ll::make_attr("uidset", uidset) << ll::make_attr("service", service)
                  << ll::make_attr("error", error) << ll::make_attr("old_cb", old_cb)
                  << ll::make_attr("new_cb", new_cb) << make_context_attr(ctx);
    }

    void deduplication_disabled(task_context_ptr ctx, const string& uid, const string& service)
    {
        S_SPECIAL << ll::make_attr("status", "deduplicate user: disabled")
                  << ll::make_attr("uid", uid) << ll::make_attr("service", service)
                  << make_context_attr(ctx);
    }

    void deduplicate_list_failed(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& error)
    {
        S_SPECIAL << ll::make_attr("status", "deduplicate user: list failed")
                  << ll::make_attr("uid", uid) << ll::make_attr("service", service)
                  << ll::make_attr("error", error) << make_context_attr(ctx);
    }

    void deduplicate_unsubscribe_failed(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& error)
    {
        S_SPECIAL << ll::make_attr("status", "deduplicate user: unsubscribe failed")
                  << ll::make_attr("uid", uid) << ll::make_attr("service", service)
                  << ll::make_attr("error", error) << make_context_attr(ctx);
    }

    void subscription_is_duplicate(
        task_context_ptr ctx,
        const sub_t& duplicate,
        const sub_t& original,
        const string& reason)
    {
        S_COMMON << ll::make_attr("status", "duplicate") << make_field_attrs(duplicate)
                 << ll::make_attr("original_id", original.id)
                 << ll::make_attr("original_device", original.device)
                 << ll::make_attr("reason", reason) << make_context_attr(ctx);
    }

    void deduplicate_unsubscribe_finished(task_context_ptr ctx, const sub_t& subscription)
    {
        S_SPECIAL << ll::make_attr("status", "deduplicate_unsubscribe finished")
                  << make_field_attrs(subscription) << make_context_attr(ctx);
    }

    void deduplicate_unsubscribe_conflict(task_context_ptr ctx, const sub_t& subscription)
    {
        S_SPECIAL << ll::make_attr("status", "deduplicate_unsubscribe conflict")
                  << make_field_attrs(subscription) << make_context_attr(ctx);
    }

private:
    void parse_notification_fields(string const& log_notification_fields_str)
    {
        if (!log_notification_fields_str.empty())
        {
            boost::split(
                log_notification_fields_,
                log_notification_fields_str,
                boost::is_any_of(","),
                boost::algorithm::token_compress_on);
        }
    }

    void log_map_fields(const std::map<string, string>& map, ll::attributes_map& am)
    {
        for (const auto& key : log_notification_fields_)
        {
            auto pair = map.find(key);
            am << ll::make_attr(key, pair != map.end() ? pair->second : "");
        }
    }

    ll::attributes_map make_attrs(const message& message)
    {
        ll::attributes_map am;
        am << ll::make_attr("uid", message.uid) << ll::make_attr("service", message.service)
           << ll::make_attr("event", message.operation);
        if (message.local_id != 0)
        {
            am << ll::make_attr("local_id", message.local_id);
        }
        am << ll::make_attr("transit_id", message.transit_id);
        am << ll::make_attr("ttl", message.ttl);
        am << ll::make_attr("payload_sz", message.raw_data.size());
        am << ll::make_attr("keys_sz", calc_keys_size(message));
        am << ll::make_attr("delivery_type", delivery_type_for(message));
        log_map_fields(message.data, am);
        return am;
    }

    ll::attributes_map make_field_attrs(
        const packet& packet,
        const std::map<string, string>& custom_fields = {})
    {
        auto delta = elapsed_time_ms(packet.message);
        string experiments = boost::algorithm::join(packet.message.experiments, ",");

        ll::attributes_map am = make_field_attrs(packet.subscription);
        am << make_context_attr(packet.ctx) << ll::make_attr("uid", packet.uid)
           << ll::make_attr("service", packet.service)
           << ll::make_attr("event", packet.message.operation)
           << ll::make_attr("bright", packet.message.bright ? "true" : "false")
           << ll::make_attr("delivery_mode", packet.robust_delivery ? "queued" : "direct")
           << ll::make_attr("delivery_type", delivery_type_for(packet.message));
        if (packet.message.local_id != 0)
        {
            am << ll::make_attr("local_id", packet.message.local_id);
        }
        am << ll::make_attr("transit_id", packet.message.transit_id)
           << ll::make_attr("event_ts", packet.message.event_ts)
           << ll::make_attr("time_elapsed", delta) << ll::make_attr("ttl", packet.message.ttl)
           << ll::make_attr("remaining_ttl", packet.ttl)
           << ll::make_attr("payload_sz", calc_payload_size(packet, custom_fields));

        for (auto&& field : custom_fields)
        {
            am << ll::make_attr(field.first, field.second);
        }

        am << ll::make_attr("experiments", experiments);
        log_map_fields(packet.message.data, am);
        return am;
    }

    ll::attributes_map make_field_attrs(const sub_t& subscription)
    {
        ll::attributes_map am;
        am << ll::make_attr("uid", subscription.uid)
           << ll::make_attr("service", subscription.service)
           << ll::make_attr("subscription_id", subscription.id)
           << ll::make_attr("platform", subscription.platform)
           << ll::make_attr("session", subscription.session_key)
           << ll::make_attr("client", subscription.client)
           << ll::make_attr("local_id", subscription.init_local_id)
           << ll::make_attr("device", subscription.device)
           << ll::make_attr("transport", get_transport(subscription));

        if (is_mobile_uri(subscription.callback_url))
        {
            string app_name;
            string push_token;
            if (parse_mobile_uri(subscription.callback_url, app_name, push_token))
            {
                am << ll::make_attr("app_name", app_name)
                   << ll::make_attr("push_token_md5", yplatform::md5(push_token));
            }
        }
        else if (is_apns_queue_uri(subscription.callback_url))
        {
            string ignored;
            string app_name;
            if (parse_apns_queue_uri(subscription.callback_url, ignored, app_name, ignored))
            {
                am << ll::make_attr("app_name", app_name);
            }
        }
        else if (is_webpush_uri(subscription.callback_url))
        {
            string origin;
            if (parse_webpush_uri_origin(subscription.callback_url, origin))
            {
                am << ll::make_attr("webpush_origin", origin);
            }
        }
        return am;
    }

    size_t calc_keys_size(const message& msg)
    {
        size_t total = 0;
        for (auto& p : msg.data)
        {
            total += p.first.size() + p.second.size();
        }
        return total;
    }

    size_t calc_payload_size(
        const packet& packet,
        const std::map<string, string>& custom_fields = {})
    {
        size_t result = packet.message.raw_data.size();
        if (packet.subscription.platform.size())
        {
            auto it = custom_fields.find("repacked_size");
            size_t replacement = 0;
            if (it != custom_fields.end() &&
                boost::conversion::try_lexical_convert<size_t>(it->second, replacement))
            {
                result = replacement;
            }
        }
        return result;
    }

    string get_transport(const sub_t& sub)
    {
        if (is_mobile_uri(sub.callback_url) || is_apns_queue_uri(sub.callback_url))
        {
            return "mobile"s;
        }
        else if (is_webpush_uri(sub.callback_url))
        {
            return "webpush"s;
        }
        else if (is_xiva_websocket_uri(sub.callback_url))
        {
            return "websocket"s;
        }
        else
        {
            return "http"s;
        }
    }

    string delivery_type_for(const message& message)
    {
        if (message.broadcast)
        {
            return message.uid.empty() ? "broadcast_in"s : "broadcast_out"s;
        }
        if (message.batch_size) return "multicast"s;
        return "unicast"s;
    }

private:
    ll::attributes_map root_am_notifications;
    ll::attributes_map root_am_notifications_special;
    ll::attributes_map root_am_subscriptions;
    ll::attributes_map root_am_subscriptions_special;
    std::vector<string> log_notification_fields_;
    boost::optional<yplatform::log::tskv_logger> logger_;
};

}}
