#pragma once

#include "cluster_metadata.h"
#include <yxiva/core/types.h>
#include <yxiva/core/iabstract.h>
#include <yxiva/core/types.h>
#include <yxiva/core/tvm.h>
#include <yxiva/core/read_text_file.h>
#include <atomic>
#include <set>

namespace yxiva { namespace hub {

struct settings_t
{
    unsigned default_ttl;
    bool ignore_xstore_errors;
    bool force_weak_delivery;
    bool deduplicate_xtable;

    time_duration subscribe_timeout;
    time_duration unsubscribe_timeout;
    time_duration unsubscribe_all_timeout;
    time_duration notify_timeout;
    time_duration batch_notify_timeout;
    time_duration send_result_timeout_delta;
    time_duration update_uidset_callback_timeout;
    time_duration finalize_delay; // to eliminate stats.finalize flaps

    unsigned message_max_tags;

    std::time_t subscribe_counters_lookup_time;

    // TODO rename to control lock name
    string service_resource_name = { "xtasks-service" };

    struct
    {
        string control_view_content;
    } api;

    tvm_checker_settings tvm;

    struct
    {
        string name;
        bool use_static_metadata = false;
        unsigned sync_port_offset = 0;
        unsigned lease_port_offset = 0;
        cluster_metadata static_metadata;
        struct
        {
            std::set<string> arbiters;
            unsigned acquire_lease_timeout = 1000;
            unsigned max_lease_time = 5000;
            bool verbose_logging = 0;
            time_duration init_delay = seconds(30);
        } leasemeta;
    } cluster;

    void read(yplatform::ptree const& conf)
    {
        default_ttl = conf.get("default_ttl", 25);
        ignore_xstore_errors = conf.get<bool>("ignore_xstore_errors", false);
        force_weak_delivery = conf.get<bool>("force_weak_delivery", 0);
        deduplicate_xtable = conf.get<bool>("deduplicate_xtable", 1);
        subscribe_timeout = conf.get<time_duration>("subscribe_timeout");
        unsubscribe_timeout = conf.get<time_duration>("unsubscribe_timeout");
        unsubscribe_all_timeout = conf.get<time_duration>("unsubscribe_all_timeout");
        notify_timeout = conf.get<time_duration>("notify_timeout");
        batch_notify_timeout = conf.get<time_duration>("batch_notify_timeout");
        send_result_timeout_delta = conf.get<time_duration>("send_result_timeout_delta");
        update_uidset_callback_timeout = conf.get<time_duration>("update_uidset_callback_timeout");

        finalize_delay =
            std::max(conf.get<time_duration>("finalize_delay"), time_duration(seconds(1)));

        message_max_tags = conf.get<unsigned>("message_max_tags", 10);

        subscribe_counters_lookup_time =
            conf.get<std::time_t>("subscribe_counters_lookup_time", 24 * 60 * 60);

        service_resource_name = conf.get("service_resource_name", service_resource_name);

        string control_view_content_file = conf.get("api.control_view_content", "");
        api.control_view_content = read_text_file(control_view_content_file);

        if (auto tvm_conf = conf.get_child_optional("tvm"))
        {
            tvm.parse_ptree(*tvm_conf);
        }

        cluster.name = conf.get("cluster.name", cluster.name);
        cluster.sync_port_offset = conf.get<unsigned>("cluster.sync_port_offset");
        cluster.lease_port_offset = conf.get<unsigned>("cluster.lease_port_offset");
        auto static_metadata_conf = conf.get_child_optional("cluster.static_metadata");
        if (static_metadata_conf)
        {
            cluster.use_static_metadata = true;
            cluster.static_metadata.parse_ptree(*static_metadata_conf);
            if (!cluster.static_metadata.valid)
            {
                throw std::domain_error(
                    "invalid cluster metadata " + cluster.static_metadata.dump());
            }
        }
        else
        {
            cluster.use_static_metadata = false;
            yplatform::read_ptree(
                cluster.leasemeta.arbiters, conf.get_child("cluster.leasemeta"), "arbiters");
            cluster.leasemeta.acquire_lease_timeout = conf.get(
                "cluster.leasemeta.acquire_lease_timeout", cluster.leasemeta.acquire_lease_timeout);
            cluster.leasemeta.max_lease_time =
                conf.get("cluster.leasemeta.max_lease_time", cluster.leasemeta.max_lease_time);
            cluster.leasemeta.verbose_logging =
                conf.get("cluster.leasemeta.verbose_logging", cluster.leasemeta.verbose_logging);
            cluster.leasemeta.init_delay = conf.get<time_duration>("cluster.leasemeta.init_delay");
        }
    }
};

typedef boost::shared_ptr<settings_t> settings_ptr;

}}