#pragma once

#include <yplatform/util/sstream.h>
#include <atomic>
#include <ctime>

namespace yxiva { namespace hub {

struct diag_sample
{
    unsigned errors = 0;
    bool no_master = 0;
};

class moving_diag_stats
{
public:
    void init(size_t capacity)
    {
        samples_.resize(capacity);
    }

    void add(const diag_sample& sample)
    {
        samples_[index_++] = sample;
        if (index_ >= samples_.size()) index_ = 0;
    }

    size_t position() const
    {
        return index_;
    }

    void reset()
    {
        for (auto& s : samples_)
        {
            s = diag_sample{};
        }
        index_ = 0;
    }

    const std::vector<diag_sample>& samples() const
    {
        return samples_;
    }

    string dump() const
    {
        std::stringstream ss;
        ss << "[";
        for (auto&& sample : samples_)
        {
            ss << "{" << sample.errors << "," << sample.no_master << "},";
        }
        ss << "]";
        return ss.str();
    }

private:
    std::vector<diag_sample> samples_;
    unsigned index_ = { 0 };
};

class module_stats
{
public:
    std::atomic<bool> control_leader = { false };
    std::atomic<bool> convey_enabled = { false };
    std::atomic<bool> robust_delivery = { 0 };
    std::atomic<bool> ignore_xstore_errors = { false };
    std::atomic<bool> deduplicate_xtable = { false };
    std::atomic<bool> manual_xtasks_degradation = { false };
    std::atomic<bool> manual_xstore_degradation = { false };
    std::atomic<unsigned> xtasks_errors = { 0 };
    moving_diag_stats self_diag;

    // Triggers application stop after a delay (settings.finalize_delay).
    std::atomic<bool> finalize{ false };

    cluster_metadata get_cluster_metadata()
    {
        scoped_lock lock(m_);
        return cluster_metadata_;
    }

    void set_cluster_metadata(const cluster_metadata& cluster_metadata)
    {
        scoped_lock lock(m_);
        cluster_metadata_ = cluster_metadata;
    }

    yplatform::ptree to_ptree() const
    {
        yplatform::ptree result;
        result.put("xtasks_errors", xtasks_errors.load());
        result.put("control_leader", control_leader.load());
        result.put("convey_enabled", convey_enabled.load());
        result.put("robust_delivery", robust_delivery.load());
        result.put("ignore_xstore_errors", ignore_xstore_errors.load());
        result.put("deduplicate_xtable", deduplicate_xtable.load());
        result.put("manual_xtasks_degradation", manual_xtasks_degradation.load());
        result.put("manual_xstore_degradation", manual_xstore_degradation.load());
        result.put("finalize", finalize.load());
        string samples;
        yplatform::sstream ss(samples, 100);
        for (auto& sample : self_diag.samples())
        {
            if (sample.no_master)
            {
                ss << "NM";
            }
            else
            {
                ss << sample.errors;
            }
            ss << " ";
        }
        result.put("self_diag.samples", samples);
        result.put("self_diag.index", self_diag.position());
        return result;
    }

private:
    mutex m_;
    cluster_metadata cluster_metadata_;
};

}}
