#pragma once

#include "coro_root.h"
#include "finalize_coro.h"
#include "xtask_context.h"
#include "worker_procedures.h"

#include <boost/asio/yield.hpp>
#define XTASK_LOG(level) YLOG_CTX(root.logger, context, level)

namespace yxiva { namespace hub { namespace worker {

struct convey_coro : public boost::asio::coroutine
{
    coro_root& root;
    xtask_context_ptr context;
    // job and subscription are owned by the context
    xtask_context::job& job;
    sub_t& subscription;

    convey_coro(
        coro_root& root,
        xtask_context_ptr context,
        xtask_context::job& job,
        sub_t& subscription)
        : root(root), context(context), job(job), subscription(subscription)
    {
    }

    struct
    {
        error_code err;
        bool exists;
        local_id_t local_id;
    } xtable_result;

    // xtable handler
    void operator()(const error_code& error, bool exists, local_id_t local_id)
    {
        xtable_result.err = error;
        xtable_result.exists = exists;
        xtable_result.local_id = local_id;
        (*this)();
    }

    void operator()(const error_code& send_err = error_code(), const string& = string())
    {
        shared_ptr<yxiva::message> msg;
        bool subscription_dropped = false;
        time_t new_subscription_retry_interval = subscription.retry_interval;
        auto now = time(nullptr);

        try
        {
            reenter(this)
            {
                increase_stats_counter();
                job.final_event_ts = subscription.ack_event_ts;

                for (; !job.finished(); job.step++)
                {

                    if (time_is_up(root.exhaust->send_timeout_for(subscription)))
                    {
                        XTASK_LOG(info) << "time is up";
                        job.recreate_task = true;
                        break;
                    }

                    // message ttl have expired
                    if (job.steps[job.step].message->event_ts + job.steps[job.step].message->ttl <
                        now)
                    {
                        job.final_id = job.steps[job.step].message->local_id;
                        job.final_event_ts = job.steps[job.step].message->event_ts;
                        context->transport_log->convey_expired(
                            make_packet(job.steps[job.step], subscription));
                        continue;
                    }

                    yield root.exhaust->send(
                        make_packet(job.steps[job.step], subscription), false, *this);
                    if (send_err.value() == err_code_subscription_dropped)
                    {
                        job.final_id = 0;
                        job.final_event_ts = 0;
                        job.retry_interval = 0;
                        job.recreate_task = false;
                        subscription_dropped = true;
                        break;
                    }

                    if (is_retriable_error(send_err))
                    {
                        // Fake a reasonable ack_local_id for all subscriptions to help
                        // with lag detection.
                        job.final_id = job.steps[job.step].message->local_id - 1;
                        new_subscription_retry_interval =
                            retry_interval_for(subscription, root.settings);
                        job.retry_interval = new_subscription_retry_interval;
                        job.recreate_task = false;
                        break;
                    }

                    job.final_id = job.steps[job.step].message->local_id;
                    job.final_event_ts = job.steps[job.step].message->event_ts;
                    new_subscription_retry_interval = 0;
                    job.retry_interval = new_subscription_retry_interval;
                }

                if (!root.progressive_retry_enabled)
                {
                    new_subscription_retry_interval = 0;
                    job.retry_interval = 0;
                }

                if (!subscription_dropped &&
                    (job.final_id > subscription.ack_local_id || job.retry_interval > 0))
                {
                    yield root.xtable->update(
                        context,
                        context->task.uid,
                        context->task.service,
                        subscription.id,
                        subscription.ack_local_id,
                        job.final_id,
                        new_subscription_retry_interval,
                        job.final_event_ts,
                        *this);

                    // retry only once
                    if (xtable_result.err)
                    {
                        context->transport_log->convey_job_update_failed(
                            context, subscription, job.final_id, xtable_result.err.message());
                        yield root.xtable->update(
                            context,
                            context->task.uid,
                            context->task.service,
                            subscription.id,
                            subscription.ack_local_id,
                            job.final_id,
                            new_subscription_retry_interval,
                            job.final_event_ts,
                            *this);
                    }

                    if (xtable_result.err)
                    {
                        context->transport_log->convey_job_update_failed(
                            context, subscription, job.final_id, xtable_result.err.message());
                        context->has_fails = true;
                    }
                    else
                    {
                        if (xtable_result.exists && xtable_result.local_id != job.final_id)
                        {
                            context->transport_log->convey_job_update_failed(
                                context,
                                subscription,
                                job.final_id,
                                "conflict server_local_id " +
                                    std::to_string(xtable_result.local_id) + " != old_local_id " +
                                    std::to_string(subscription.ack_local_id) +
                                    " new_local_id=" + std::to_string(job.final_id));
                        }
                        else
                        {
                            context->transport_log->convey_job_update_success(
                                context, subscription, job.final_id);
                        }
                    }
                }
            }
        }
        catch (const std::exception& e)
        {
            XTASK_LOG(error) << "convey_coroutine exception: " << e.what();
            context->has_fails = true;
        }

        if (is_complete())
        {
            decrease_stats_counter();
            // last job finished
            if (--context->jobs_count == 0)
            {
                if (!context->has_fails)
                {
                    finalize_coro finalize(root, context);
                    finalize();
                }
                else
                {
                    // drop task if this or another job has errors
                    --root.total_tasks;
                }
            }
        }
    }

    void increase_stats_counter()
    {
        if (job.retry)
        {
            ++root.total_retry_jobs_convey;
        }
        else
        {
            ++root.total_jobs;
        }
    }

    void decrease_stats_counter()
    {
        if (job.retry)
        {
            --root.total_retry_jobs_convey;
        }
        else
        {
            --root.total_jobs;
        }
    }

    bool time_is_up(const time_duration& send_timeout)
    {
        return clock::now() + send_timeout >= context->deadline;
    }

    ::yxiva::packet make_packet(const xtask_context::send_step& step, const sub_t& subscription)
    {
        auto ret = ::yxiva::packet(context, *step.message, subscription);
        ret.mark_queued_delivery();
        if (step.action == filter::action::send_silent)
        {
            ret.mark_silent_delivery();
        }
        return ret;
    }

    bool is_retriable_error(const error_code& err)
    {
        switch (err.value())
        {
        case err_code_no_error:
        // Supposed to be processed before, repeated here to be on the safe side.
        case err_code_subscription_dropped:
        case err_code_agent_bad_request:
        case err_code_forbidden:
        case err_code_unprocessable:
            return false;
        default:
            return true;
        }
    }
};

}}}

#undef XTASK_LOG
#include <boost/asio/unyield.hpp>
