#pragma once

#include "coro_root.h"
#include "xtask_context.h"
#include <algorithm>
#include <limits>

#include <boost/asio/yield.hpp>
#define XTASK_LOG(level) YLOG_CTX(root.logger, context, level)

namespace yxiva { namespace hub { namespace worker {

namespace {
inline bool need_recreate_task(const xtask_context& c)
{
    using job = xtask_context::job;
    return std::any_of(
        c.jobs.begin(), c.jobs.end(), [](const job& j) -> bool { return j.recreate_task; });
}

inline local_id_t recreate_local_id(const xtask_context& c)
{
    local_id_t min_final_id = std::numeric_limits<local_id_t>::max();
    for (auto& job : c.jobs)
    {
        if (job.recreate_task && job.final_id < min_final_id)
        {
            min_final_id = job.final_id;
        }
    }
    return min_final_id + 1;
}
}

struct finalize_coro : public boost::asio::coroutine
{
    finalize_coro(coro_root& root, xtask_context_ptr context) : root(root), context(context)
    {
    }

    void operator()(const ymod_xtasks::error& xtasks_err = ymod_xtasks::error())
    {
        time_t delay_sec;
        ymod_xtasks::task_draft draft;
        time_t task_retry_interval;

        try
        {
            reenter(this)
            {
                if (need_recreate_task(*context))
                {
                    draft = { context->task.uid,
                              context->task.service,
                              recreate_local_id(*context),
                              "",
                              ymod_xtasks::delay_flags::none };
                    yield root.xtasks->create_task(context, draft, *this);
                    if (xtasks_err)
                    {
                        XTASK_LOG(error)
                            << "create_task failed: "
                            << " uid=" << context->task.uid << " service=" << context->task.service
                            << " error=\"" << xtasks_err.code.message() << " "
                            << xtasks_err.ext_reason << "\"";
                    }
                    else
                    {
                        XTASK_LOG(info)
                            << "create_task finished: "
                            << " uid=" << context->task.uid << " service=" << context->task.service;
                    }
                }

                task_retry_interval = compute_retry_interval();
                if (context->delay_task)
                {
                    delay_sec = root.settings.default_delay;
                    XTASK_LOG(debug) << "delay_task delay_sec=" << delay_sec;
                    yield root.xtasks->delay_task(
                        context, root.name, context->task.id, delay_sec, *this);
                    if (xtasks_err)
                    {
                        XTASK_LOG(error) << "delay_task error=" << xtasks_err.code.message()
                                         << " ext_reason=" << xtasks_err.ext_reason;
                    }
                    else
                    {
                        XTASK_LOG(info)
                            << "xtask delayed: total_time="
                            << time_traits::to_string(clock::now() - context->start_time)
                            << "delay_sec=" << delay_sec;
                    }
                }
                else if (task_retry_interval > 0)
                {
                    XTASK_LOG(debug) << "delay_task_shallow delay_sec=" << task_retry_interval;
                    yield root.xtasks->delay_task_shallow(
                        context, root.name, context->task.id, task_retry_interval, *this);
                    if (xtasks_err)
                    {
                        XTASK_LOG(error) << "delay_task_shallow error=" << xtasks_err.code.message()
                                         << " ext_reason=" << xtasks_err.ext_reason;
                    }
                    else
                    {
                        XTASK_LOG(info)
                            << "xtask retry: total_time="
                            << time_traits::to_string(clock::now() - context->start_time)
                            << "task_retry_interval=" << task_retry_interval;
                    }
                }
                else
                {
                    yield root.xtasks->fin_task(context, root.name, context->task.id, *this);
                    if (xtasks_err)
                    {
                        XTASK_LOG(error) << "fin_task error=" << xtasks_err.code.message()
                                         << " ext_reason=" << xtasks_err.ext_reason;
                    }
                    else
                    {
                        XTASK_LOG(info)
                            << "xtask finished: total_time="
                            << time_traits::to_string(clock::now() - context->start_time);
                    }
                }
            }
        }
        catch (const std::exception& e)
        {
            XTASK_LOG(error) << "finalize_coroutine exception: " << e.what();
        }

        if (is_complete())
        {
            --root.total_tasks;
        }
    }

    time_t compute_retry_interval()
    {
        if (!root.progressive_retry_enabled)
        {
            return 0;
        }
        auto min_interval = context->filtered_retry_interval;
        for (auto& job : context->jobs)
        {
            if (min_interval == 0)
            {
                min_interval = job.retry_interval;
                continue;
            }
            if (job.retry_interval > 0 && job.retry_interval < min_interval)
            {
                min_interval = job.retry_interval;
            }
        }
        return min_interval;
    }

    coro_root& root;
    xtask_context_ptr context;
};

}}}

#undef XTASK_LOG
#include <boost/asio/unyield.hpp>
