#pragma once

#include "coro_root.h"
#include "retry_coro.h"
#include "convey_coro.h"
#include "finalize_coro.h"
#include "worker_procedures.h"
#include "xtask_context.h"

#include <boost/asio/yield.hpp>
#define XTASK_LOG(level) YLOG_CTX(root.logger, context, level)

namespace yxiva { namespace hub { namespace worker {

namespace {
inline bool less(shared_ptr<message> const& m1, shared_ptr<message> const& m2)
{
    return m1->local_id < m2->local_id;
}
}

struct launch_coro : public boost::asio::coroutine
{
    launch_coro(coro_root& root, xtask_context_ptr context, unsigned generation)
        : root(root), context(context), generation_(generation)
    {
        context->start_time = clock::now();
        context->deadline =
            context->start_time + root.settings.kill_timeout - root.settings.xtable_update_timeout;
    }

    // xtable handler
    void operator()(const error_code& err, const sub_list& list)
    {
        xtable_result.err = err;
        // don't store in context to prevent unhandled exceptions
        if (!err) xtable_result.list = &list;
        (*this)();
    }

    // xstore handler
    void operator()(const ymod_xstore::error& err, ymod_xstore::entries_list_ptr entries)
    {
        xstore_result.err = err;
        if (!err) xstore_result.entries = entries;
        (*this)();
    }

    void operator()()
    {
        bool task_complete = false;

        try
        {
            reenter(this)
            {
                ++root.total_tasks; // once

                yield root.xtable->find(
                    context, context->task.uid, context->task.service, {}, *this);

                if (xtable_result.err)
                {
                    XTASK_LOG(error) << "xtable list failed: " << xtable_result.err.message();
                    break;
                }

                if (root.expired(generation_))
                {
                    XTASK_LOG(notice) << "shutdown";
                    break;
                }

                if (xtable_result.list->empty())
                {
                    XTASK_LOG(info) << "xtable list: no subscriptions";
                    task_complete = true;
                    break;
                }

                context->subscriptions = *xtable_result.list;
                log_list_debug();

                if (root.progressive_retry_enabled)
                {
                    filter_future_retries();
                    if (context->subscriptions.empty())
                    {
                        XTASK_LOG(info) << "filter_future_retries: no subscriptions";
                        task_complete = true;
                        break;
                    }
                    log_list_debug();

                    filter_lagging_subscriptions();
                    log_list_debug();
                }

                //   if (calc_min_id() == 0) break;

                if (context->subscriptions.size())
                {
                    yield root.xstore->read(
                        context,
                        user_id(context->task.uid),
                        context->task.service,
                        calc_min_id(),
                        root.settings.xstore_read_limit,
                        0,
                        0,
                        *this);

                    if (root.expired(generation_))
                    {
                        XTASK_LOG(notice) << "shutdown";
                        break;
                    }

                    if (xstore_result.err || xstore_result.entries->empty())
                    {
                        if (xstore_result.err)
                        {
                            XTASK_LOG(error)
                                << "read xstore: error=" << xstore_result.err.code.message()
                                << " ext_reason=" << xstore_result.err.ext_reason;
                        }
                        else
                        {
                            XTASK_LOG(info) << "read xstore: no messages";
                        }
                        task_complete = context->subscriptions_for_retry.empty();
                    }
                    else
                    {
                        decode_messages(xstore_result.entries, context->messages);
                        xstore_result.entries.reset();

                        prepare_jobs(context, root.logger);
                    }
                }

                // Retry jobs will be prepared in retry_coros.
                for (auto& s : context->subscriptions_for_retry)
                {
                    context->jobs.emplace_back(s);
                    context->jobs.back().retry = true;
                }

                context->jobs_count += static_cast<unsigned>(context->jobs.size());
                for (auto& job : context->jobs)
                {
                    auto& subscription = job.subscription;
                    if (job.retry)
                    {
                        XTASK_LOG(info) << "spawning retry coro:"
                                        << " subscription=" << subscription.id
                                        << " client=" << subscription.client;
                        yplatform::spawn(
                            std::make_shared<retry_coro>(root, context, job, generation_));
                    }
                    else
                    {
                        job.recreate_task =
                            context->messages.size() >= root.settings.xstore_read_limit;
                        XTASK_LOG(info) << "starting job:"
                                        << " subscription=" << subscription.id
                                        << " client=" << subscription.client << " steps=["
                                        << job.steps_to_string() << "]";
                        convey_coro convey(root, context, job, subscription);
                        convey();
                    }
                }
            }
        }
        catch (const std::exception& e)
        {
            context->has_fails = true;
            XTASK_LOG(error) << "launch_coroutine exception: " << e.what();
        }

        if (is_complete())
        {
            if (task_complete)
            {
                finalize_coro finalize(root, context);
                finalize();
            }
            else if (context->jobs.empty())
            {
                --root.total_tasks;
            }
        }
    }

    local_id_t find_max_ack_local_id(const sub_list& subs)
    {
        local_id_t max = 0;
        for (auto& s : subs)
        {
            if (s.retry_interval == 0 && s.ack_local_id > max)
            {
                max = s.ack_local_id;
            }
        }
        return max;
    }

    void filter_future_retries()
    {
        auto now = time(nullptr);
        context->subscriptions.erase(
            std::remove_if(
                context->subscriptions.begin(),
                context->subscriptions.end(),
                [now, this](const sub_t& s) -> bool {
                    if (s.next_retry_time > now)
                    {
                        auto retry_in = s.next_retry_time - now;
                        if (context->filtered_retry_interval == 0 ||
                            context->filtered_retry_interval > retry_in)
                            context->filtered_retry_interval = retry_in;
                    }
                    return s.next_retry_time > now;
                }),
            context->subscriptions.end());
    }

    void filter_lagging_subscriptions()
    {
        auto max_id = find_max_ack_local_id(context->subscriptions);
        auto max_diff = root.settings.xstore_read_limit;
        context->subscriptions.erase(
            std::remove_if(
                context->subscriptions.begin(),
                context->subscriptions.end(),
                [max_id, max_diff, context = context](const sub_t& s) -> bool {
                    bool ret = s.ack_local_id != 0 ? max_id - s.ack_local_id > max_diff :
                                                     s.retry_interval > 0;

                    if (ret)
                    {
                        context->subscriptions_for_retry.push_back(s);
                    }
                    return ret;
                }),
            context->subscriptions.end());
    }

    void log_list_debug()
    {
        for (auto& i : context->subscriptions)
        {
            XTASK_LOG(debug) << "DEBUG LIST id=" << i.id << " client=" << i.client
                             << " ack_local_id=" << i.ack_local_id << " ack_time=" << i.ack_time
                             << " retry_interval=" << i.retry_interval
                             << " next_retry_time=" << i.next_retry_time;
        }
    }

    // find min local_id (not 0)
    local_id_t calc_min_id()
    {
        auto min_id = context->subscriptions.front().ack_local_id;
        if (min_id != 0) min_id++;
        for (auto& s : context->subscriptions)
        {
            if (s.ack_local_id != 0 && s.ack_local_id < min_id)
            {
                min_id = s.ack_local_id + 1;
            }
        }
        if (min_id == 0)
        {
            min_id = context->task.local_id;
        }
        XTASK_LOG(debug) << "process_list min_id=" << min_id
                         << " task.local_id=" << context->task.local_id;
        return min_id;
    }

    coro_root& root;
    xtask_context_ptr context;

    struct
    {
        error_code err;
        const sub_list* list = nullptr;
    } xtable_result;

    struct
    {
        ymod_xstore::error err;
        ymod_xstore::entries_list_ptr entries;
    } xstore_result;
    unsigned generation_;
};

}}}

#undef XTASK_LOG
#include <boost/asio/unyield.hpp>
