#pragma once

#include "coro_root.h"
#include "launch_coro.h"
#include "finalize_coro.h"
#include "worker_procedures.h"
#include "xtask_context.h"
#include <yplatform/context_repository.h>
#include <yplatform/task_context.h>

#include <boost/asio/yield.hpp>

namespace yxiva { namespace hub { namespace worker {

// spawns child coroutines:
// launch -> convey -> finalize
struct main_loop_coro : public boost::asio::coroutine
{
    // use own task_context to store in context_repo
    main_loop_coro(coro_root& root, unsigned generation)
        : root(root), main_loop_context(boost::make_shared<task_context>()), generation_(generation)
    {
    }

    // xtasks handler
    void operator()(const ymod_xtasks::error& err, const std::vector<ymod_xtasks::task>& tasks)
    {
        xtasks_result.err = err;
        xtasks_result.tasks = &tasks;
        (*this)();
    }

    void operator()(const error_code& /*ec*/ = error_code())
    {
        unsigned tasks_limit;

        try
        {
            reenter(this)
            {
                YLOG(root.logger, info) << "main_coro generation " << generation_ << " started";
                context_repo.add_context(main_loop_context);

                for (;;)
                {

                    // if stop, wait unitl all tasks are completed
                    do
                    {
                        root.exec_timer->expires_from_now(root.settings.exec_interval);
                        yield root.exec_timer->async_wait(*this);
                    } while (root.expired(generation_) && root.total_tasks);

                    if (root.expired(generation_))
                    {
                        break;
                    }

                    tasks_limit = calc_tasks_limit();
                    if (tasks_limit)
                    {
                        yield root.xtasks->get_tasks(
                            main_loop_context, root.name, tasks_limit, *this);
                        if (!xtasks_result.err)
                        {
                            launch_xtasks();
                        }
                        else
                        {
                            log_xtasks_failed();
                        }
                    }
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG(root.logger, error)
                << "main_coro generation " << generation_ << " exception: " << e.what();
        }

        if (is_complete())
        {
            assert(root.total_tasks == 0);
            context_repo.rem_context(main_loop_context);
            YLOG(root.logger, info) << "main_coro generation " << generation_ << " complete";
        }
    }

    unsigned calc_tasks_limit()
    {
        unsigned settings_count = root.settings.max_tasks;
        if (root.lazy_mode) settings_count /= root.settings.lazy_factor;
        return (settings_count > root.total_tasks &&
                settings_count - root.total_tasks >= settings_count / 3) ?
            (settings_count - root.total_tasks) :
            0U;
    }

    void log_xtasks_failed()
    {
        YLOG(root.logger, error) << "get_tasks failed: error=" << xtasks_result.err.code.message()
                                 << " ext_reason=" << xtasks_result.err.ext_reason;
    }

    void launch_xtasks()
    {
        if (xtasks_result.tasks->size())
        {
            YLOG(root.logger, debug) << "get_tasks ok: count=" << xtasks_result.tasks->size();
            for (auto& task : *xtasks_result.tasks)
            {
                launch_coro launch(
                    root, boost::make_shared<xtask_context>(task, root.transport_log), generation_);
                launch();
            }
        }
    }

    coro_root& root;
    yplatform::task_context_ptr main_loop_context;
    struct
    {
        // err is stored by value to prevent implicitly
        // wrong if-statements like 'if (err)'
        ymod_xtasks::error err;
        const std::vector<ymod_xtasks::task>* tasks = nullptr;
    } xtasks_result;
    unsigned generation_;
};

struct alive_coro : public boost::asio::coroutine
{
    alive_coro(coro_root& root, unsigned generation)
        : root(root), context(boost::make_shared<task_context>()), generation_(generation)
    {
    }

    void operator()(const error_code& /*ec*/ = error_code())
    {
        try
        {
            reenter(this)
            {
                YLOG(root.logger, info) << "alive_coro generation " << generation_ << " started";
                while (!root.expired(generation_))
                {
                    root.alive_timer->expires_from_now(root.settings.alive_interval);
                    yield root.alive_timer->async_wait(*this);
                    if (root.expired(generation_)) break;

                    yield root.xtasks->alive(context, root.name, *this);
                    if (xtasks_error)
                    {
                        YLOG(root.logger, error) << "alive failed: " << xtasks_error.code.message()
                                                 << " " << xtasks_error.ext_reason;
                    }
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG(root.logger, error)
                << "alive_coro generation " << generation_ << " exception: " << e.what();
        }

        if (is_complete())
        {
            YLOG(root.logger, info) << "alive_coro generation " << generation_ << " complete";
        }
    }

    // xtasks handler
    void operator()(const ymod_xtasks::error& err)
    {
        xtasks_error = err;
        (*this)();
    }

    coro_root& root;
    yplatform::task_context_ptr context;
    ymod_xtasks::error xtasks_error;
    unsigned generation_;
};

}}}

#include <boost/asio/unyield.hpp>