#pragma once

#include "coro_root.h"
#include "convey_coro.h"
#include "finalize_coro.h"
#include "worker_procedures.h"
#include "xtask_context.h"
#include <yplatform/coroutine.h>

#include <boost/asio/yield.hpp>
#define XTASK_LOG(level) YLOG_CTX(root.logger, context, level)

namespace yxiva { namespace hub { namespace worker {

struct retry_coro
{
    using yield_context_t = yplatform::yield_context<retry_coro>;

    coro_root& root;
    xtask_context_ptr context;
    // subscription is owned by the context
    sub_t& subscription;

    xtask_context::job& job;
    std::vector<shared_ptr<yxiva::message>> messages;
    unsigned generation_;

    retry_coro(
        coro_root& root,
        xtask_context_ptr context,
        xtask_context::job& job,
        unsigned generation)
        : root(root)
        , context(context)
        , subscription(job.subscription)
        , job(job)
        , generation_(generation)
    {
    }

    struct
    {
        ymod_xstore::error err;
        ymod_xstore::entries_list_ptr entries;
    } xstore_result;

    // xstore handler
    void operator()(
        yield_context_t yield_context,
        const ymod_xstore::error& err,
        ymod_xstore::entries_list_ptr entries)
    {
        xstore_result.err = err;
        if (!err) xstore_result.entries = entries;
        (*this)(yield_context);
    }

    void operator()(yield_context_t yield_context)
    {
        time_t begin_ts;
        time_t end_ts;

        try
        {
            reenter(yield_context)
            {
                ++root.total_retry_jobs_xstore;

                // NB: this query might take a while if the subscription was not
                // responding for a long time.
                begin_ts = subscription.ack_event_ts > 0 ? subscription.ack_event_ts :
                                                           subscription.init_time;
                end_ts = time(nullptr) + 60 * 60;

                yield root.xstore->read(
                    context,
                    user_id(context->task.uid),
                    context->task.service,
                    subscription.ack_local_id,
                    root.settings.xstore_read_limit,
                    begin_ts,
                    end_ts,
                    yield_context);

                if (root.expired(generation_))
                {
                    XTASK_LOG(notice) << "shutdown";
                    yield break;
                }

                if (xstore_result.err)
                {
                    XTASK_LOG(error) << "read xstore: error=" << xstore_result.err.code.message()
                                     << " ext_reason=" << xstore_result.err.ext_reason;
                    job.final_id = subscription.ack_local_id;
                    job.retry_interval = retry_interval_for(subscription, root.settings);
                    subscription.retry_interval = job.retry_interval;
                }
                else if (xstore_result.entries->empty())
                {
                    XTASK_LOG(info) << "read xstore: no messages for lagging subscription";
                    job.retry_interval = 0;
                    yield break;
                }
                else
                {
                    decode_messages(xstore_result.entries, messages);
                    xstore_result.entries.reset();

                    prepare_job(context, root.logger, messages, job);
                }

                ++context->jobs_count;
                XTASK_LOG(info) << "starting job (retry): subscription=" << subscription.id
                                << " client=" << subscription.client << " steps=["
                                << job.steps_to_string() << "]";
                job.recreate_task = messages.size() >= root.settings.xstore_read_limit;
                convey_coro convey(root, context, job, subscription);
                convey();
            }
        }
        catch (const std::exception& e)
        {
            XTASK_LOG(error) << "retry_coroutine exception: " << e.what();
            context->has_fails = true;
        }

        if (yield_context.is_complete())
        {
            --root.total_retry_jobs_xstore;
            // last job finished
            if (--context->jobs_count == 0)
            {
                if (!context->has_fails)
                {
                    finalize_coro finalize(root, context);
                    finalize();
                }
                else
                {
                    // drop task if this or another job has errors
                    --root.total_tasks;
                }
            }
        }
    }
};

}}}

#undef XTASK_LOG
#include <boost/asio/unyield.hpp>
