#include "worker_procedures.h"

#include <yxiva/core/filter.h>
#include <yxiva/core/callbacks.h>
#include <algorithm>

namespace yxiva { namespace hub { namespace worker {

bool message_relevant_for(const message& message, const sub_t& sub);

void decode_messages(
    const ymod_xstore::entries_list_ptr& entries,
    std::vector<shared_ptr<yxiva::message>>& messages)
{
    messages.reserve(entries->size());

    for (auto& entry : *entries)
    {
        auto msg = make_shared<yxiva::message>();
        unpack(entry.content, *msg);
        msg->local_id = entry.local_id;
        msg->event_ts = entry.event_ts;
        messages.push_back(msg);
    }

    std::sort(
        messages.begin(),
        messages.end(),
        [](shared_ptr<message> const& m1, shared_ptr<message> const& m2) -> bool {
            return m1->local_id < m2->local_id;
        });
}

void prepare_jobs(xtask_context_ptr context, yplatform::log::source& logger)
{
    assert(context->messages.size() != 0);
    assert(context->subscriptions.size() != 0);
    assert(context->jobs.size() == 0);

    for (size_t i = 0; i < context->subscriptions.size(); ++i)
    {
        auto& subscription = context->subscriptions[i];
        xtask_context::job job(subscription);

        if (!prepare_job(context, logger, context->messages, job))
        {
            continue;
        }

        context->jobs.push_back(std::move(job));
    }
}

bool prepare_job(
    const xtask_context_ptr& ctx,
    yplatform::log::source& logger,
    const std::vector<shared_ptr<yxiva::message>>& messages,
    xtask_context::job& job)
{
    const auto& subscription = job.subscription;
    local_id_t final_id = subscription.ack_local_id;

    if (messages.back()->local_id <= subscription.ack_local_id)
    {
        return false;
    }

    job.steps.clear();
    for (size_t j = 0; j < messages.size(); ++j)
    {
        auto message = messages[j];
        if (!message_relevant_for(*message, subscription))
        {
            struct packet packet(ctx, *message, subscription);
            ctx->transport_log->convey_skip(packet, "too old");
            final_id = message->local_id;
            continue;
        }
        if (message->local_id <= subscription.ack_local_id) continue;
        ::yxiva::filter::action action;
        if (auto parsed = filter::parse_and_apply(action, subscription, *message))
        {
            if (action == filter::action::skip)
            {
                struct packet packet(ctx, *message, subscription);
                packet.mark_queued_delivery();
                ctx->transport_log->convey_skip_filtered(packet);
                final_id = message->local_id;
                continue;
            }
        }
        else
        {
            YLOG(logger, info) << "filter parse error=" << parsed.error_reason
                               << " ctx=" << ctx->uniq_id();
        }

        xtask_context::send_step step = { message, action };
        job.steps.push_back(step);
    }

    if (job.steps.size())
    {
        // Final ID will be incremented after each step is successfully completed.
        job.final_id = subscription.ack_local_id;

        YLOG(logger, info) << "prepare_jobs new job"
                           << " subscription.id=" << subscription.id
                           << " subscription.ack_local_id=" << subscription.ack_local_id
                           << " min_id=" << job.steps.front().message->local_id
                           << " max_id=" << job.steps.back().message->local_id
                           << " final_id=" << job.final_id << " retry=" << job.retry
                           << " ctx=" << ctx->uniq_id();
    }
    else
    {
        job.final_id = final_id;

        YLOG(logger, info) << "prepare_jobs new update-only job"
                           << " subscription.id=" << subscription.id
                           << " subscription.ack_local_id=" << subscription.ack_local_id
                           << " final_id=" << job.final_id << " retry=" << job.retry
                           << " ctx=" << ctx->uniq_id();
    }

    return true;
}

bool message_relevant_for(const message& message, const sub_t& sub)
{
    static const int MESSAGE_NOT_OLDER_SECONDS = 2;
    auto delta = message.event_ts - sub.init_time;
    return sub.ack_local_id != 0 || delta > -MESSAGE_NOT_OLDER_SECONDS;
}

time_t retry_interval_for(const sub_t& subscription, const settings& settings)
{
    const settings::retry& retry = retry_for(subscription, settings);
    if (subscription.retry_interval == 0)
    {
        return retry.start_interval;
    }

    time_t interval = subscription.retry_interval * retry.backoff_coefficient;
    return interval < retry.end_interval ? interval : retry.end_interval;
}

const settings::retry& retry_for(const sub_t& subscription, const settings& settings)
{
    if (callback_uri::is_mobile_uri(subscription.callback_url))
    {
        return settings.retry_mobile;
    }
    else if (callback_uri::is_webpush_uri(subscription.callback_url))
    {
        return settings.retry_webpush;
    }
    else
    {
        return settings.retry_http;
    }
}

}}}
