#pragma once

#include <ymod_xtasks/xtasks.h>
#include <yxiva/core/types.h>
#include <sintimers/queue.h>
#include <types.h>

namespace yxiva { namespace hub {

class xtasks_service
    : public std::enable_shared_from_this<xtasks_service>
    , public yplatform::log::contains_logger
{
public:
    struct settings
    {
        boost::shared_ptr<ymod_xtasks::xtasks> xtasks;
        time_duration cleanup_interval = seconds(10);
        time_duration cleanup_workers_interval = seconds(5);
        time_duration wakeup_interval = seconds(5);
        unsigned task_exec_sec = 20;
        unsigned worker_alive_sec = 5;

        void parse_ptree(const yplatform::ptree& conf)
        {
            cleanup_interval = conf.get("cleanup_interval", cleanup_interval);
            cleanup_workers_interval =
                conf.get("cleanup_workers_interval", cleanup_workers_interval);
            wakeup_interval = conf.get("wakeup_interval", wakeup_interval);
            task_exec_sec = time_traits::get_seconds_count(
                conf.get<time_duration>("task_exec", seconds(task_exec_sec)));
            worker_alive_sec = time_traits::get_seconds_count(
                conf.get<time_duration>("worker_alive", seconds(worker_alive_sec)));
        }
    };

    typedef shared_ptr<settings> settings_ptr;

    xtasks_service(timers::queue_ptr timers, settings_ptr st)
    {
        context_ = boost::make_shared<yplatform::task_context>();
        timers_ = timers;
        stop_ = true;
        settings_ = st;
    }

    virtual void start()
    {
        if (!stop_) return;
        stop_ = false;

        cleanup_timer_ = timers_->create_timer();
        cleanup_workers_timer_ = timers_->create_timer();
        wakeup_timer_ = timers_->create_timer();

        cleanup_timer_start();
        cleanup_workers_timer_start();
        wakeup_timer_start();
    }

    virtual void stop()
    {
        if (stop_) return;
        stop_ = true;

        cleanup_timer_->cancel();
        cleanup_workers_timer_->cancel();
        wakeup_timer_->cancel();
    }

    settings_ptr get_settings()
    {
        scoped_lock lock(mutex_);
        return settings_;
    }

    void update_settings(settings_ptr st)
    {
        scoped_lock lock(mutex_);
        settings_ = st;
    }

    virtual ~xtasks_service()
    {
    }

private:
    void cleanup_timer_start()
    {
        if (stop_) return;
        cleanup_timer_->async_wait(
            get_settings()->cleanup_interval,
            std::bind(&xtasks_service::cleanup, shared_from_this()));
    }

    void cleanup_workers_timer_start()
    {
        if (stop_) return;
        cleanup_workers_timer_->async_wait(
            get_settings()->cleanup_workers_interval,
            std::bind(&xtasks_service::cleanup_workers, shared_from_this()));
    }

    void wakeup_timer_start()
    {
        if (stop_) return;
        wakeup_timer_->async_wait(
            get_settings()->wakeup_interval,
            std::bind(&xtasks_service::wakeup, shared_from_this()));
    }

    void cleanup()
    {
        if (stop_) return;
        auto settings = get_settings();
        auto self = shared_from_this();
        settings->xtasks->cleanup_active(
            context_,
            settings->task_exec_sec,
            [this, self](const ymod_xtasks::error& err, const size_t count) {
                if (err)
                {
                    YLOG_L(error) << "xtasks cleanup active failed " << err.code.message() << " "
                                  << err.ext_reason;
                }
                else
                {
                    if (count) YLOG_L(info) << "xtasks cleanup active " << count << " tasks";
                }
                cleanup_timer_start();
            });
    }

    void cleanup_workers()
    {
        if (stop_) return;
        auto settings = get_settings();
        auto self = shared_from_this();
        settings->xtasks->cleanup_workers(
            context_,
            settings->worker_alive_sec,
            [this, self](const ymod_xtasks::error& err, const size_t count) {
                if (err)
                {
                    YLOG_L(error) << "xtasks cleanup workers failed " << err.code.message() << " "
                                  << err.ext_reason;
                }
                else
                {
                    if (count) YLOG_L(info) << "xtasks cleanup workers " << count << " tasks";
                }
                cleanup_workers_timer_start();
            });
    }

    void wakeup()
    {
        if (stop_) return;
        auto settings = get_settings();
        auto self = shared_from_this();
        settings->xtasks->wakeup_delayed(
            context_, [this, self](const ymod_xtasks::error& err, const size_t count) {
                if (err)
                {
                    YLOG_L(error) << "xtasks wakeup failed " << err.code.message() << " "
                                  << err.ext_reason;
                }
                else
                {
                    if (count) YLOG_L(info) << "xtasks waked up " << count << " tasks";
                }
                wakeup_timer_start();
            });
    }

    yplatform::task_context_ptr context_;
    mutex mutex_;
    bool stop_;
    timers::queue_ptr timers_;
    settings_ptr settings_;
    timers::timer_ptr cleanup_timer_;
    timers::timer_ptr cleanup_workers_timer_;
    timers::timer_ptr wakeup_timer_;
};

}}
