#pragma once

#include "xtable.h"

#include "istorage.h"
#include "pq_storage.h"
#include "memory_storage.h"
#include "../error.h"
#include <yxiva/core/iabstract.h>
#include <yxiva/core/types.h>
#include <yxiva/core/message.h>
#include <yxiva/core/split.h>
#include <yxiva/core/callbacks.h> // gate detection in hacks::
#include <yxiva/core/subscription_id.h>
#include <yxiva/core/shards/storage.h>
#include <yplatform/module.h>
#include <yplatform/find.h>
#include <yplatform/util/safe_call.h>
#include <boost/uuid/string_generator.hpp>
#include <boost/uuid/uuid_io.hpp>
#include <boost/system/error_code.hpp>
#include <algorithm>

namespace yxiva { namespace hub {

struct SubscribeContext
{
    task_context_ptr ctx;
    sub_t subscription;

    error_code error;
    subscribe_callback_t callback;
};

struct UnsubscribeContext
{
    task_context_ptr ctx;
    string subscription_id;
    string uid;
    string service;

    error_code error;
    callback_t callback;
};

namespace hacks {

inline void remove_inactive_subscriptions(sub_list& list)
{
    list.erase(
        std::remove_if(
            list.begin(),
            list.end(),
            [](const sub_t& sub) -> bool {
                return !callback_uri::is_active_uri(sub.callback_url);
            }),
        list.end());
}

}

inline void filter_find_results(sub_list& list, const XTable::find_options& options)
{
    if (!options.show_inactive)
    {
        hacks::remove_inactive_subscriptions(list);
    }
}

class xtable_impl
    : public XTable
    , public yplatform::module
{
    shared_ptr<IStorage> subscriptions_storage;
    std::shared_ptr<shard_config::storage> shards_;
    std::optional<db_role> forced_db_role_;

    typedef xtable_impl this_t;
    typedef std::function<void(this_t*, const error_code&, const user_info&)> auth_callback_t;

public:
    void init(shared_ptr<IStorage> storage)
    {
        subscriptions_storage = storage;
    }

    void init(yplatform::ptree const& conf)
    {
        string mod_pq_xtable_name = conf.get("pq_xtable", "pq_xtable");
        auto pq_xivadb = yplatform::find<ymod_pq::cluster_call>(mod_pq_xtable_name);
        string mod_pq_batch_name = conf.get("pq_batch", "pq_batch");
        auto pq_batch = yplatform::find<ymod_pq::cluster_call>(mod_pq_batch_name);
        string shards_name = conf.get("shards", "shards_xtable");
        shards_ = yplatform::find<shard_config::storage, std::shared_ptr>(shards_name);
        subscriptions_storage = boost::make_shared<PQStorage>(pq_xivadb, pq_batch, shards_);
    }

    void force_db_role(std::optional<db_role> t) override
    {
        forced_db_role_ = t;
    }

    void subscribe_mobile(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const filter_t& filter,
        const string& callback_url,
        const string& extra_data,
        const string& client,
        const string& device_uuid,
        const ttl_t ttl,
        const string& subscription_id,
        const local_id_t local_id,
        const string& platform,
        const string& device,
        const string& bb_connection_id,
        const subscribe_callback_t& callback) override
    {
        sub_t subscription;
        subscription.uid = uid;
        subscription.service = service;
        subscription.filter = filter;
        subscription.callback_url = callback_url;
        subscription.extra_data = extra_data;
        subscription.client = client;
        subscription.session_key = canonize_device_id(device_uuid);
        subscription.ttl = ttl;
        subscription.init_local_id = local_id;
        subscription.ack_local_id = local_id;
        subscription.platform = platform::hacks::fcm_to_gcm(platform); // gcm_compatibility
        subscription.device = device;
        subscription.bb_connection_id = bb_connection_id;
        subscription.id =
            subscription_id.empty() ? make_mobile_subscription_id(device_uuid) : subscription_id;

        subscribe(ctx, subscription, callback);
    }

    void subscribe(
        task_context_ptr ctx,
        const sub_t& subscription,
        const subscribe_callback_t& callback) override
    {
        auto call_context = make_shared<SubscribeContext>();
        call_context->ctx = ctx;
        call_context->subscription = subscription;
        if (call_context->subscription.id.empty())
        {
            call_context->subscription.id = make_subscription_id(call_context->subscription);
        }
        call_context->callback = callback;
        start_subscribe(call_context);
    }

    void start_subscribe(shared_ptr<SubscribeContext> context)
    {
        subscriptions_storage->add(
            context->ctx,
            context->subscription,
            boost::bind(&this_t::handle_storage_add, shared_from(this), _1, context));
    }

    void handle_storage_add(const error_code& error, shared_ptr<SubscribeContext> context)
    {
        if (error)
        {
            YLOG_CTX_LOCAL(context->ctx, error) << "storage add failed: " << error.message();
            context->error = error;
            context->callback(context->error, context->subscription);
            return;
        }

        YLOG_CTX_LOCAL(context->ctx, info)
            << "storage add finished:"
            << " subscription-id=\"" << context->subscription.id << "\" uid=\""
            << context->subscription.uid << "\" service=\"" << context->subscription.service
            << "\" client=\"" << context->subscription.client << "\" callback_url=\""
            << context->subscription.callback_url << "\" ttl=\"" << context->subscription.ttl
            << "\"";

        if (context->ctx->is_cancelled())
        {
            context->callback(make_error_code(err_code::err_code_cancelled), context->subscription);
            return;
        }

        context->callback(error_code(), context->subscription);
    }

    void handle_storage_del(const error_code& error, shared_ptr<UnsubscribeContext> context)
    {
        if (error)
        {
            YLOG_CTX_LOCAL(context->ctx, error) << "storage del failed: " << error.message();
            context->error = error;
        }
        else
        {
            YLOG_CTX_LOCAL(context->ctx, info) << "storage del finished";
        }

        if (context->callback) context->callback(context->error, context->subscription_id);
    }

    void unsubscribe(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& subscription_id,
        const callback_t& callback) override
    {
        auto context = make_shared<UnsubscribeContext>();
        context->ctx = ctx;
        context->uid = uid;
        context->service = service;
        context->subscription_id = subscription_id;
        context->callback = callback;

        subscriptions_storage->del(
            context->ctx,
            context->uid,
            context->service,
            context->subscription_id,
            boost::bind(&this_t::handle_storage_del, shared_from(this), _1, context));
    }

    void unsubscribe_mobile(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& device_uuid,
        const callback_t& callback) override
    {
        unsubscribe(ctx, uid, service, make_mobile_subscription_id(device_uuid), callback);
    }

    void unsubscribe_overlapped(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& subscription_id,
        const unsigned overlap_sec,
        const callback_t& callback) override
    {
        auto context = make_shared<UnsubscribeContext>();
        context->ctx = ctx;
        context->uid = uid;
        context->service = service;
        context->subscription_id = subscription_id;
        context->callback = callback;

        subscriptions_storage->del_overlapped(
            context->ctx,
            context->uid,
            context->service,
            context->subscription_id,
            overlap_sec,
            boost::bind(&this_t::handle_storage_del, shared_from(this), _1, context));
    }

    void batch_unsubscribe(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const std::vector<string>& subscription_ids,
        std::time_t init_not_after,
        const batch_del_callback_t& callback) override
    {
        subscriptions_storage->batch_del(
            ctx,
            uid,
            service,
            subscription_ids,
            init_not_after,
            [this, ctx, callback, self = shared_from(this)](
                const error_code& ec, const std::vector<string>& unsubscribed_ids) {
                if (ec)
                {
                    YLOG_CTX_LOCAL(ctx, error) << "storage batch_del failed: " << ec.message();
                }
                else
                {
                    YLOG_CTX_LOCAL(ctx, info) << "storage batch_del finished";
                }
                yplatform::safe_call(
                    ctx, "batch_unsubscribe callback", callback, ec, unsubscribed_ids);
            });
    }

    void find(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const find_options& options,
        const find_callback_t& callback) override
    {
        auto db_role = forced_db_role_ ? *forced_db_role_ : options.db_role;
        subscriptions_storage->find(
            ctx, uid, service, db_role, [callback, options](const error_code& ec, sub_list list) {
                filter_find_results(list, options);
                callback(ec, std::move(list));
            });
    }

    void batch_find(
        task_context_ptr ctx,
        const batch_keys& keys,
        const string& service,
        const find_options& options,
        const batch_find_callback_t& callback) override
    {
        auto db_role = forced_db_role_ ? *forced_db_role_ : options.db_role;
        subscriptions_storage->batch_find(
            ctx,
            keys,
            service,
            db_role,
            [callback,
             options](const error_code& ec, const batch_iterators& key_iterators, sub_list list) {
                filter_find_results(list, options);
                callback(ec, key_iterators, std::move(list));
            });
    }

    void update(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& subscription_id,
        local_id_t old_local_id,
        local_id_t new_local_id,
        time_t retry_interval,
        time_t ack_event_ts,
        const update_callback_t& callback) override
    {
        subscriptions_storage->update(
            ctx,
            uid,
            service,
            subscription_id,
            old_local_id,
            new_local_id,
            retry_interval,
            ack_event_ts,
            callback);
    }

    void add_broken_subscription(
        task_context_ptr ctx,
        const string& platform,
        const string& subscription_id,
        std::time_t timestamp,
        const add_callback_t& callback) override
    {
        subscriptions_storage->add_broken_subscription(
            ctx,
            platform::hacks::fcm_to_gcm(platform),
            subscription_id,
            timestamp,
            callback); // gcm_compatibility
    }

    void find_uidset(
        task_context_ptr ctx,
        const string& uidset,
        const string& service,
        const find_options& options,
        const find_callback_t& callback) override
    {
        auto db_role = forced_db_role_ ? *forced_db_role_ : options.db_role;
        subscriptions_storage->find_uidset(
            ctx,
            uidset,
            service,
            db_role,
            [callback, options](const error_code& ec, sub_list list) {
                filter_find_results(list, options);
                callback(ec, std::move(list));
            });
    }

    void update_uidset(
        task_context_ptr ctx,
        const string& uidset,
        const string& service,
        const string& old_cb,
        const string& new_cb,
        const update_uidset_callback_t& callback) override
    {
        subscriptions_storage->update_uidset(ctx, uidset, service, old_cb, new_cb, callback);
    }

    void update_callback(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& subscription_id,
        const string& old_cb,
        const string& new_cb,
        const update_uidset_callback_t& callback) override
    {
        subscriptions_storage->update_callback(
            ctx, uid, service, subscription_id, old_cb, new_cb, callback);
    }

    void batch_find(
        task_context_ptr ctx,
        const string& uid,
        const std::vector<string>& services,
        const find_options& options,
        const find_callback_t& callback) override
    {
        auto db_role = forced_db_role_ ? *forced_db_role_ : options.db_role;
        subscriptions_storage->batch_find(
            ctx, uid, services, db_role, [callback, options](const error_code& ec, sub_list list) {
                filter_find_results(list, options);
                callback(ec, std::move(list));
            });
    }

    std::shared_ptr<shard_config::storage> shards() override
    {
        return shards_;
    }

    virtual void enable_fallback(task_context_ptr ctx) override
    {
        return subscriptions_storage->enable_fallback(ctx);
    }

    virtual void disable_fallback(task_context_ptr ctx) override
    {
        return subscriptions_storage->disable_fallback(ctx);
    }
};

}}
