#pragma once

#include <boost/uuid/string_generator.hpp>
#include <boost/uuid/uuid_io.hpp>
#include <boost/system/error_code.hpp>

#include <ymod_pq/call.h>
#include <yxiva/core/iabstract.h>
#include <yxiva/core/types.h>
#include <yxiva/core/authorizer.h>
#include <yxiva/core/message.h>
#include <yxiva/core/subscriptions.h>
#include <yxiva/core/batch_key.h>
#include <yxiva/core/shards/shards.h>

#include <yplatform/hash/sha1.h>
#include "../error.h"

namespace yxiva { namespace hub {

typedef string operation_t;
typedef string filter_t;
typedef std::vector<batch_keys::const_iterator> batch_iterators;

typedef std::function<void(const error_code&)> add_callback_t;
typedef std::function<void(const error_code&)> del_callback_t;
typedef std::function<void(const error_code&, const std::vector<string>&)> batch_del_callback_t;
typedef std::function<void(const error_code&, sub_list)> find_callback_t;
typedef std::function<void(const error_code&, const batch_iterators&, sub_list)>
    batch_find_callback_t;
typedef std::function<void(const error_code&, bool updated, local_id_t)> update_callback_t;
typedef std::function<void(const error_code&)> update_uidset_callback_t;

class IStorage : public iabstract
{
public:
    enum class db_role
    {
        master,
        replica
    };

    virtual void add(
        task_context_ptr ctx,
        const sub_t& subscription,
        const add_callback_t& callback) = 0;
    virtual void del(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& subscription_id,
        const del_callback_t& callback) = 0;
    virtual void del_overlapped(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& subscription_id,
        const unsigned overlap_sec,
        const del_callback_t& callback) = 0;
    virtual void batch_del(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const std::vector<string>& subscription_ids,
        std::time_t init_not_after,
        const batch_del_callback_t& callback) = 0;
    virtual void find(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        db_role target,
        const find_callback_t& callback) = 0;
    virtual void batch_find(
        task_context_ptr ctx,
        const batch_keys& keys,
        const string& service,
        db_role target,
        const batch_find_callback_t& callback) = 0;
    virtual void update(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& subscription_id,
        local_id_t old_local_id,
        local_id_t new_local_id,
        time_t retry_interval,
        time_t ack_event_ts,
        const update_callback_t& callback) = 0;
    virtual void add_broken_subscription(
        task_context_ptr ctx,
        const string& platform,
        const string& subscription_id,
        std::time_t timestamp,
        const add_callback_t& callback) = 0;
    virtual void find_uidset(
        task_context_ptr ctx,
        const string& uidset,
        const string& service,
        db_role target,
        const find_callback_t& callback) = 0;
    virtual void update_uidset(
        task_context_ptr ctx,
        const string& uidset,
        const string& service,
        const string& old_cb,
        const string& new_cb,
        const update_uidset_callback_t& callback) = 0;
    virtual void update_callback(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& subscription_id,
        const string& old_cb,
        const string& new_cb,
        const update_uidset_callback_t& callback) = 0;
    virtual void batch_find(
        task_context_ptr ctx,
        const string& uid,
        const std::vector<string>& services,
        db_role target,
        const find_callback_t& callback) = 0;
    virtual void enable_fallback(task_context_ptr ctx) = 0;
    virtual void disable_fallback(task_context_ptr ctx) = 0;
};

}}
