#pragma once

#include "istorage.h"

namespace yxiva { namespace hub {

class MemoryStorage : public IStorage
{
    std::map<root_params_t, std::vector<sub_t>> list;

public:
    void add(task_context_ptr /*ctx*/, const sub_t& subscription, const add_callback_t& callback)
        override
    {
        auto& record = list[subscription];

        bool exist = false;
        for (auto it = record.begin(); it != record.end(); ++it)
        {
            if (it->id == subscription.id)
            {
                it->init_local_id = 0;
                it->init_time = std::time(0);
                it->ack_local_id = 0;
                it->ack_time = std::time(0);
                it->client = subscription.client;
                it->session_key = subscription.session_key;
                it->callback_url = subscription.callback_url;
                it->platform = subscription.platform;
                exist = true;
                break;
            }
        }

        if (!exist)
        {
            auto copy = subscription;
            copy.init_local_id = 0;
            copy.init_time = std::time(0);
            copy.ack_local_id = 0;
            copy.ack_time = std::time(0);
            record.push_back(copy);
        }
        callback(error_code());
    }

    void del(
        task_context_ptr /*ctx*/,
        const string& uid,
        const string& /*service*/,
        const string& subscription_id,
        const del_callback_t& callback) override
    {
        for (auto it = list.begin(); it != list.end(); ++it)
        {
            if (it->first.uid == static_cast<string>(uid))
            {
                for (size_t i = 0; i < it->second.size(); ++i)
                {
                    if (it->second[i].id == subscription_id)
                    {
                        it->second.erase(it->second.begin() + i);
                        callback(error_code());
                        return;
                    }
                }
            }
        }
        callback(error_code());
    }

    void del_overlapped(
        task_context_ptr /*ctx*/,
        const string& uid,
        const string& /*service*/,
        const string& subscription_id,
        const unsigned overlap_sec,
        const del_callback_t& callback) override
    {
        for (auto it = list.begin(); it != list.end(); ++it)
        {
            if (it->first.uid == static_cast<string>(uid))
            {
                for (size_t i = 0; i < it->second.size(); ++i)
                {
                    if (it->second[i].id == subscription_id &&
                        it->second[i].init_time >= std::time(0) + overlap_sec)
                    {
                        it->second.erase(it->second.begin() + i);
                        callback(error_code());
                        return;
                    }
                }
            }
        }
        callback(error_code());
    }

    void find(
        task_context_ptr /*ctx*/,
        const string& uid,
        const string& service,
        db_role /*db_role*/,
        const find_callback_t& callback) override
    {
        if (list.empty())
        {
            callback(error_code(), std::vector<sub_t>());
            return;
        }
        for (auto it = list.begin(); it != list.end(); ++it)
        {
            if (it->first.uid == static_cast<string>(uid) &&
                it->first.service == static_cast<string>(service))
            {
                callback(error_code(), it->second);
            }
        }
    }

    void batch_find(
        task_context_ptr /*ctx*/,
        const batch_keys& keys,
        const string& service,
        db_role /*db_role*/,
        const batch_find_callback_t& callback) override
    {
        batch_iterators iterators;
        for (auto i_key = keys.begin(); i_key != keys.end(); ++i_key)
        {
            iterators.push_back(i_key);
        }

        if (list.empty())
        {
            callback(error_code(), iterators, std::vector<sub_t>());
            return;
        }

        sub_list result;

        for (const auto& key : keys)
        {
            auto i_root_sub_pair = list.find(root_params_t{ key.uid, service });
            if (i_root_sub_pair == list.end())
            {
                continue;
            }

            const auto& stored_subs = i_root_sub_pair->second;

            if (key.subscription_id.empty())
            {
                for (const auto& sub : stored_subs)
                {
                    result.push_back(sub);
                }
            }
            else
            {
                auto i_desired_sub = std::find_if(
                    stored_subs.begin(), stored_subs.end(), [&key](const sub_t& s) -> bool {
                        return s.id == key.subscription_id;
                    });
                if (i_desired_sub != stored_subs.end())
                {
                    result.push_back(*i_desired_sub);
                }
            }
        }

        callback(error_code(), iterators, result);
    }

    void update(
        task_context_ptr /*ctx*/,
        const string& uid,
        const string& service,
        const string& subscription_id,
        local_id_t old_local_id,
        local_id_t new_local_id,
        time_t retry_interval,
        time_t,
        const update_callback_t& callback) override
    {
        for (auto it = list.begin(); it != list.end(); ++it)
        {
            if (it->first.uid != static_cast<string>(uid) ||
                it->first.service != static_cast<string>(service))
            {
                continue;
            }

            for (size_t i = 0; i < it->second.size(); ++i)
            {
                if (it->second[i].id == subscription_id)
                {
                    if (old_local_id == it->second[i].ack_local_id)
                    {
                        it->second[i].ack_local_id = new_local_id;
                        it->second[i].retry_interval = retry_interval;
                        callback(error_code(), true, new_local_id);
                    }
                    else
                    {
                        callback(error_code(), true, it->second[i].ack_local_id);
                    }
                }
            }
        }
        callback(error_code(), false, 0);
    }

    void add_broken_subscription(
        task_context_ptr /*ctx*/,
        const string& /*platform*/,
        const string& /*subscription_id*/,
        std::time_t /*timestamp*/,
        const add_callback_t& callback) override
    {
        callback(error_code());
    }

    void find_uidset(
        task_context_ptr /*ctx*/,
        const string& /*uidset*/,
        const string& /*service*/,
        db_role /*db_role*/,
        const find_callback_t& /*callback*/) override
    {
        throw std::runtime_error("implement me if you need me");
    }

    void update_uidset(
        task_context_ptr /*ctx*/,
        const string& /*uidset*/,
        const string& /*service*/,
        const string& /*old_cb*/,
        const string& /*new_cb*/,
        const update_uidset_callback_t& /*callback*/) override
    {
        throw std::runtime_error("implement me if you need me");
    }

    void update_callback(
        task_context_ptr /*ctx*/,
        const string& /*uid*/,
        const string& /*service*/,
        const string& /*subscription_id*/,
        const string& /*old_cb*/,
        const string& /*new_cb*/,
        const update_uidset_callback_t& /*callback*/) override
    {
        throw std::runtime_error("implement me if you need me");
    }

    void batch_find(
        task_context_ptr /*ctx*/,
        const string& /*uid*/,
        const std::vector<string>& /*services*/,
        db_role /*db_role*/,
        const find_callback_t& /*callback*/) override
    {
        throw std::runtime_error("implement me if you need me");
    }

    virtual void batch_del(
        task_context_ptr /*ctx*/,
        const string& /*uid*/,
        const string& /*service*/,
        const std::vector<string>& /*subscription_ids*/,
        std::time_t /*ts*/,
        const batch_del_callback_t& /*callback*/) override
    {
        throw std::runtime_error("implement me if you need me");
    }

    void enable_fallback(task_context_ptr) override
    {
        throw;
    }
    void disable_fallback(task_context_ptr) override
    {
        throw;
    }
};

}}
