#pragma once

#include "pq/list.h"
#include "pq/list_no_acks.h"
#include "pq/batch_list.h"
#include "pq/dummy.h"
#include "pq/update.h"
#include "pq/batch_del.h"
#include "istorage.h"
#include <ymod_pq/call.h>
#include <yxiva/core/shards/storage.h>
#include <yxiva/core/gid.h>
#include <yplatform/log.h>
#include <boost/bind/protect.hpp>
#include <boost/date_time/posix_time/posix_time.hpp> // for literal timestamps
#include <unordered_map>

#define SAFE_CALL(where, expr)                                                                     \
    try                                                                                            \
    {                                                                                              \
        (expr);                                                                                    \
    }                                                                                              \
    catch (const std::exception& e)                                                                \
    {                                                                                              \
        YLOG_CTX_GLOBAL(ctx, error) << where << " callback exception=" << e.what();                \
    }                                                                                              \
    catch (...)                                                                                    \
    {                                                                                              \
        YLOG_CTX_GLOBAL(ctx, error) << where << " callback exception=unknown";                     \
    }

namespace yxiva { namespace hub {
namespace {

// reorders iterators to put all batch keys with
// non-empty subscription ids first. Needed because
// apq does not support sparse arrays
class pq_ordered_batch
{
public:
    // iterators in order in which they were added
    const batch_iterators& iterators() const
    {
        return iterators_;
    }

    // iterators in order required by ymod_pq/apq
    const batch_iterators& reordered_iterators() const
    {
        return reordered_iterators_;
    }

    void reserve(std::size_t total_keys)
    {
        iterators_.reserve(total_keys);
        single_uid_iterators_.reserve(total_keys);
        uid_with_sub_iterators_.reserve(total_keys);
    }

    void reorder_iterators()
    {
        uid_with_sub_iterators_.swap(reordered_iterators_);
        reordered_iterators_.insert(
            reordered_iterators_.end(), single_uid_iterators_.begin(), single_uid_iterators_.end());
    }

    void add(const batch_keys::const_iterator& i_key)
    {
        if (!reordered_iterators_.empty())
        {
            throw std::runtime_error(
                "pq_ordered_batch: add() called when reordered_iterators_ not empty!");
        }

        iterators_.push_back(i_key);
        if (i_key->subscription_id.empty())
        {
            single_uid_iterators_.push_back(i_key);
        }
        else
        {
            uid_with_sub_iterators_.push_back(i_key);
        }
    }

private:
    batch_iterators iterators_;
    batch_iterators reordered_iterators_;

    batch_iterators single_uid_iterators_;
    batch_iterators uid_with_sub_iterators_;
};

inline void unpack_batch(
    const batch_keys& keys,
    const batch_iterators& iterators,
    std::vector<string>& out_uids,
    std::vector<string>& out_desired_subscription_ids)
{
    out_uids.reserve(keys.size());
    out_desired_subscription_ids.reserve(keys.size());

    for (const auto& i_key : iterators)
    {
        out_uids.push_back(i_key->uid);
        if (!i_key->subscription_id.empty())
        {
            out_desired_subscription_ids.push_back(i_key->subscription_id);
        }
    }
}

template <typename F, typename C>
void handle_call_result(task_context_ptr ctx, F future, const C& callback)
{
    try
    {
        future.get();
    }
    catch (const std::exception& e)
    {
        YLOG_CTX_GLOBAL(ctx, error) << "storage fail: " << e.what();
        SAFE_CALL("handle_call_result", callback(make_error_code(err_code_storage_fail)))
        return;
    }
    SAFE_CALL("handle_call_result", callback(error_code()))
}

template <typename F, typename C, typename H>
void handle_find_result(task_context_ptr ctx, F future, const C& callback, shared_ptr<H> list)
{
    try
    {
        future.get();
    }
    catch (const std::exception& e)
    {
        YLOG_CTX_GLOBAL(ctx, error) << "storage fail: " << e.what();
        SAFE_CALL(
            "handle_find_result", callback(make_error_code(err_code_storage_fail), sub_list()));
        return;
    }
    SAFE_CALL("handle_find_result", callback(error_code(), list->move_data()));
}

template <typename F, typename C>
void handle_batch_find_result(
    task_context_ptr ctx,
    F future,
    const C& callback,
    shared_ptr<pq_handlers::batch_list> list,
    const batch_iterators& iterators)
{
    try
    {
        future.get();
    }
    catch (const std::exception& e)
    {
        YLOG_CTX_GLOBAL(ctx, error) << "storage fail: " << e.what();
        SAFE_CALL(
            "handle_batch_find_result",
            callback(make_error_code(err_code_storage_fail), iterators, sub_list()));
        return;
    }
    SAFE_CALL("handle_batch_find_result", callback(error_code(), iterators, list->move_data()));
}

template <typename F, typename C>
void handle_update_result(
    task_context_ptr ctx,
    F future,
    const C& callback,
    shared_ptr<pq_handlers::update> update)
{
    try
    {
        future.get();
    }
    catch (const std::exception& e)
    {
        YLOG_CTX_GLOBAL(ctx, error) << "storage fail: " << e.what();
        SAFE_CALL(
            "handle_update_result", callback(make_error_code(err_code_storage_fail), false, 0));
        return;
    }
    SAFE_CALL(
        "handle_update_result", callback(error_code(), update->row_exists(), update->local_id()));
}
}

class PQStorage : public IStorage
{
    typedef boost::shared_ptr<ymod_pq::cluster_call> mod_pq_type;
    typedef std::shared_ptr<shard_config::storage> mod_shards_type;
    static constexpr yplatform::time_traits::duration req_deadline =
        yplatform::time_traits::duration::max();

public:
    PQStorage(mod_pq_type pq, mod_pq_type batch_pq, mod_shards_type shards)
        : pq_(pq), batch_pq_(batch_pq), shards_(shards)
    {
    }

    void add(task_context_ptr ctx, const sub_t& subscription, const add_callback_t& callback)
        override
    {
        auto gid = gid_from_uid(subscription.uid);
        if (subscription.uidset.size())
        {
            auto uidset_gid = gid_from_uid(subscription.uidset);
            if (gid != uidset_gid)
            {
                SAFE_CALL(
                    "subscribe_uidset",
                    callback(make_error_code(err_code_invalid_uidset_uid_shard)));
                return;
            }
        }
        auto args = boost::make_shared<ymod_pq::bind_array>();
        //        ymod_pq::push_const_int(args,
        //                                boost::lexical_cast<int>(
        //                                static_cast<string> (subscription.uid)));
        ymod_pq::push_const_string(args, static_cast<string>(subscription.uid));
        ymod_pq::push_const_string(args, std::to_string(gid));
        ymod_pq::push_const_string(args, subscription.service);
        ymod_pq::push_const_string(args, subscription.id);
        ymod_pq::push_const_string(args, subscription.callback_url);
        ymod_pq::push_const_string(args, subscription.filter);
        ymod_pq::push_const_string(args, subscription.extra_data);
        ymod_pq::push_const_string(args, subscription.client);
        ymod_pq::push_const_uint(args, subscription.ttl);
        ymod_pq::push_const_string(args, subscription.session_key);
        ymod_pq::push_const_string(args, std::to_string(subscription.init_local_id));
        if (subscription.platform.size())
        {
            ymod_pq::push_const_string(args, subscription.platform);
        }
        else
        {
            ymod_pq::push_null(args, ymod_pq::bind_array::STRING);
        }
        ymod_pq::push_const_string(args, subscription.device);
        if (subscription.bb_connection_id.size())
        {
            ymod_pq::push_const_string(args, subscription.bb_connection_id);
        }
        else
        {
            ymod_pq::push_null(args, ymod_pq::bind_array::STRING);
        }
        if (subscription.uidset.size())
        {
            ymod_pq::push_const_string(args, subscription.uidset);
        }
        else
        {
            ymod_pq::push_null(args, ymod_pq::bind_array::STRING);
        }
        auto handler = boost::make_shared<pq_handlers::dummy>();
        auto shards = shards_->get();
        auto master = master_from_gid(*shards, gid);
        if (!master)
        {
            YLOG_CTX_GLOBAL(ctx, error)
                << "failed to get shard conninfo for uid " << subscription.uid;
            SAFE_CALL("add", callback(make_error_code(err_code_storage_fail)));
            return;
        }
        ymod_pq::future_result fres = pq_->request(
            ctx,
            master->conninfo,
            "add_subscription",
            args,
            handler,
            true,
            req_deadline,
            ymod_pq::request_target::master);
        fres.add_callback(boost::bind(
            handle_call_result<decltype(fres), decltype(callback)>, ctx, fres, callback));
    }

    void del(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& subscription_id,
        const del_callback_t& callback) override
    {
        auto args = boost::make_shared<ymod_pq::bind_array>();
        ymod_pq::push_const_string(args, static_cast<string>(uid));
        ymod_pq::push_const_string(args, static_cast<string>(service));
        ymod_pq::push_const_string(args, subscription_id);
        auto handler = boost::make_shared<pq_handlers::dummy>();
        auto shards = shards_->get();
        auto master = master_from_uid(*shards, uid);
        if (!master)
        {
            YLOG_CTX_GLOBAL(ctx, error) << "failed to get shard conninfo for uid " << uid;
            SAFE_CALL("del", callback(make_error_code(err_code_storage_fail)));
            return;
        }
        ymod_pq::future_result fres = pq_->request(
            ctx,
            master->conninfo,
            "del_subscription",
            args,
            handler,
            true,
            req_deadline,
            ymod_pq::request_target::master);
        fres.add_callback(boost::bind(
            handle_call_result<decltype(fres), decltype(callback)>, ctx, fres, callback));
    }

    void del_overlapped(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& subscription_id,
        const unsigned overlap_sec,
        const del_callback_t& callback) override
    {
        auto args = boost::make_shared<ymod_pq::bind_array>();
        ymod_pq::push_const_string(args, static_cast<string>(uid));
        ymod_pq::push_const_string(args, static_cast<string>(service));
        ymod_pq::push_const_string(args, subscription_id);
        ymod_pq::push_const_uint(args, overlap_sec);
        auto handler = boost::make_shared<pq_handlers::dummy>();
        auto shards = shards_->get();
        auto master = master_from_uid(*shards, uid);
        if (!master)
        {
            YLOG_CTX_GLOBAL(ctx, error) << "failed to get shard conninfo for uid " << uid;
            SAFE_CALL("del_overlapped", callback(make_error_code(err_code_storage_fail)));
            return;
        }
        ymod_pq::future_result fres = pq_->request(
            ctx,
            master->conninfo,
            "del_subscription_overlapped",
            args,
            handler,
            true,
            req_deadline,
            ymod_pq::request_target::master);
        fres.add_callback(boost::bind(
            handle_call_result<decltype(fres), decltype(callback)>, ctx, fres, callback));
    }

    void batch_del(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const std::vector<string>& subscription_ids,
        std::time_t init_not_after,
        const batch_del_callback_t& callback) override
    {
        auto args = boost::make_shared<ymod_pq::bind_array>();
        ymod_pq::push_const_string(args, uid);
        ymod_pq::push_const_string(args, service);
        ymod_pq::push_const_svector(args, subscription_ids);
        ymod_pq::push_const_time_t(args, init_not_after);

        auto handler = boost::make_shared<pq_handlers::batch_del>();
        auto shards = shards_->get();
        auto master = master_from_uid(*shards, uid);
        if (!master)
        {
            YLOG_CTX_GLOBAL(ctx, error) << "failed to get shard conninfo for uid " << uid;
            SAFE_CALL("batch_del", callback(make_error_code(err_code_storage_fail), {}));
            return;
        }

        ymod_pq::future_result fres = batch_pq_->request(
            ctx,
            master->conninfo,
            "batch_del_subscriptions",
            args,
            handler,
            true,
            req_deadline,
            ymod_pq::request_target::master);
        fres.add_callback([ctx, fres, callback, handler]() {
            try
            {
                fres.get();
            }
            catch (const std::exception& e)
            {
                YLOG_CTX_GLOBAL(ctx, error) << "storage fail: " << e.what();
                SAFE_CALL(
                    "handle_call_result", callback(make_error_code(err_code_storage_fail), {}));
                return;
            }
            SAFE_CALL("handle_call_result", callback(error_code(), handler->get_data()))
        });
    }

    void find(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        db_role db_role,
        const find_callback_t& callback) override
    {
        auto args = boost::make_shared<ymod_pq::bind_array>();
        ymod_pq::push_const_string(args, uid);
        ymod_pq::push_const_string(args, service);

        auto handler = boost::make_shared<pq_handlers::list>(uid, service);
        auto shards = shards_->get();
        auto master = master_from_uid(*shards, uid);
        if (!master)
        {
            YLOG_CTX_GLOBAL(ctx, error) << "failed to get shard conninfo for uid " << uid;
            SAFE_CALL("find", callback(make_error_code(err_code_storage_fail), sub_list()));
            return;
        }
        ymod_pq::future_result fres = pq_->request(
            ctx,
            master->conninfo,
            "list_subscriptions",
            args,
            handler,
            true,
            req_deadline,
            target(db_role));
        fres.add_callback(boost::bind(
            handle_find_result<decltype(fres), decltype(callback), pq_handlers::list>,
            ctx,
            fres,
            callback,
            handler));
    }

    void batch_find(
        task_context_ptr ctx,
        const batch_keys& keys,
        const string& service,
        db_role db_role,
        const batch_find_callback_t& callback) override
    {
        std::unordered_map<yxiva::shard_config::id_t, pq_ordered_batch> batches_by_shard_id;

        // Id mapping does not change often, it is pointless to
        // query shards_ to get it inside the loop.
        auto shards = shards_->get();
        for (auto i_key = keys.begin(); i_key != keys.end(); ++i_key)
        {
            auto shard = shard_from_uid(*shards, i_key->uid);
            if (!shard)
            {
                YLOG_CTX_GLOBAL(ctx, error) << "failed to get shard for uid " << i_key->uid;
                continue;
            }

            auto& batch = batches_by_shard_id[shard->id];
            if (batch.iterators().empty()) batch.reserve(keys.size());

            batch.add(i_key);
        }

        for (auto& id_batch_pair : batches_by_shard_id)
        {
            const auto& shard_id = id_batch_pair.first;
            auto& batch = id_batch_pair.second;

            batch.reorder_iterators();

            auto args = boost::make_shared<ymod_pq::bind_array>();
            {
                std::vector<string> uids;
                std::vector<string> desired_sub_ids;

                unpack_batch(keys, batch.reordered_iterators(), uids, desired_sub_ids);

                ymod_pq::push_swap_svector(args, uids);
                ymod_pq::push_swap_svector(args, desired_sub_ids);
            }
            ymod_pq::push_const_string(args, service);

            auto row_handler = boost::make_shared<pq_handlers::batch_list>(service);
            auto shard = shard_from_id(*shards, shard_id);
            if (!shard)
            {
                YLOG_CTX_GLOBAL(ctx, error)
                    << "failed to get shard with id " << std::to_string(shard_id);
                SAFE_CALL(
                    "batch_find",
                    callback(
                        make_error_code(err_code_storage_fail), batch.iterators(), sub_list()));
                continue;
            }
            ymod_pq::future_result fres = batch_pq_->request(
                ctx,
                shard->master.conninfo,
                "batch_list_subscriptions",
                args,
                row_handler,
                true,
                req_deadline,
                target(db_role));
            fres.add_callback(boost::bind(
                handle_batch_find_result<decltype(fres), decltype(callback)>,
                ctx,
                fres,
                callback,
                row_handler,
                batch.iterators()));
        }
    }

    void update(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& subscription_id,
        local_id_t old_local_id,
        local_id_t new_local_id,
        time_t retry_interval,
        time_t ack_event_ts,
        const update_callback_t& callback) override
    {
        auto args = boost::make_shared<ymod_pq::bind_array>();
        ymod_pq::push_const_string(args, uid);
        ymod_pq::push_const_string(args, service);
        ymod_pq::push_const_string(args, subscription_id);
        ymod_pq::push_const_string(args, std::to_string(old_local_id));
        ymod_pq::push_const_string(args, std::to_string(new_local_id));
        ymod_pq::push_const_int(args, 0); // smart_notify - unused
        if (retry_interval > 0)
        {
            auto retry_interval_str = std::to_string(retry_interval);
            ymod_pq::push_swap_string(args, retry_interval_str);
        }
        else
        {
            ymod_pq::push_null(args, ymod_pq::bind_array::STRING);
        }
        if (ack_event_ts > 0)
        {
            ymod_pq::push_const_time_t(args, ack_event_ts);
        }
        else
        {
            ymod_pq::push_null(args, ymod_pq::bind_array::STRING);
        }

        auto handler = boost::make_shared<pq_handlers::update>();
        auto shards = shards_->get();
        auto master = master_from_uid(*shards, uid);
        if (!master)
        {
            YLOG_CTX_GLOBAL(ctx, error) << "failed to get shard conninfo for uid " << uid;
            SAFE_CALL("update", callback(make_error_code(err_code_storage_fail), false, 0));
            return;
        }
        ymod_pq::future_result fres = pq_->request(
            ctx,
            master->conninfo,
            "update_ack",
            args,
            handler,
            true,
            req_deadline,
            ymod_pq::request_target::master);
        fres.add_callback(boost::bind(
            handle_update_result<decltype(fres), decltype(callback)>,
            ctx,
            fres,
            callback,
            handler));
    }

    void add_broken_subscription(
        task_context_ptr ctx,
        const string& platform,
        const string& subscription_id,
        std::time_t timestamp,
        const add_callback_t& callback) override
    {
        string timestamp_literal = "{" +
            boost::posix_time::to_iso_string(boost::posix_time::from_time_t(timestamp)) + "+0000}";
        std::vector<string> subscription_ids{ subscription_id };

        auto args = boost::make_shared<ymod_pq::bind_array>();
        ymod_pq::push_const_string(args, platform);
        ymod_pq::push_swap_svector(args, subscription_ids);
        ymod_pq::push_swap_string(args, timestamp_literal);

        struct db_task_context
        {
            ::yxiva::mutex mutex;
            std::size_t shards_left{ 0 };
            bool done{ false };
        };

        auto shards = shards_->get();
        auto db_task_ctx = std::make_shared<db_task_context>();
        db_task_ctx->shards_left = shards->size();
        for (auto& shard : *shards)
        {
            auto handler = boost::make_shared<pq_handlers::dummy>();
            auto single_shard_cb =
                [ctx, callback, handler, db_task_ctx](const boost::system::error_code& ec) {
                    scoped_lock guard(db_task_ctx->mutex);
                    if (!db_task_ctx->done)
                    {
                        db_task_ctx->done = ec || --db_task_ctx->shards_left == 0;

                        if (db_task_ctx->done)
                        {
                            guard.unlock();
                            callback(ec);
                        }
                    }
                };

            ymod_pq::future_result fres = batch_pq_->request(
                ctx,
                shard.master.conninfo,
                "add_broken_subscription",
                args,
                handler,
                true,
                req_deadline,
                ymod_pq::request_target::master);
            fres.add_callback(boost::bind(
                handle_call_result<decltype(fres), decltype(single_shard_cb)>,
                ctx,
                fres,
                single_shard_cb));
        }
    }

    void find_uidset(
        task_context_ptr ctx,
        const string& uidset,
        const string& service,
        db_role db_role,
        const find_callback_t& callback) override
    {
        auto args = boost::make_shared<ymod_pq::bind_array>();
        ymod_pq::push_const_string(args, service);
        ymod_pq::push_const_string(args, uidset);

        auto handler = boost::make_shared<pq_handlers::list>(service_name(service));
        auto shards = shards_->get();
        auto master = master_from_uid(*shards, uidset);
        if (!master)
        {
            YLOG_CTX_GLOBAL(ctx, error) << "failed to get shard conninfo for uidset " << uidset;
            SAFE_CALL("find_uidset", callback(make_error_code(err_code_storage_fail), sub_list()));
            return;
        }
        ymod_pq::future_result fres = pq_->request(
            ctx,
            master->conninfo,
            "list_subscriptions_uidset",
            args,
            handler,
            true,
            req_deadline,
            target(db_role));
        fres.add_callback(boost::bind(
            handle_find_result<decltype(fres), decltype(callback), pq_handlers::list>,
            ctx,
            fres,
            callback,
            handler));
    }

    void update_uidset(
        task_context_ptr ctx,
        const string& uidset,
        const string& service,
        const string& old_cb,
        const string& new_cb,
        const update_uidset_callback_t& callback) override
    {
        auto args = boost::make_shared<ymod_pq::bind_array>();
        ymod_pq::push_const_string(args, uidset);
        ymod_pq::push_const_string(args, static_cast<const string&>(service));
        if (old_cb.size())
        {
            ymod_pq::push_const_string(args, old_cb);
        }
        else
        {
            ymod_pq::push_null(args, ymod_pq::bind_array::STRING);
        }
        ymod_pq::push_const_string(args, new_cb);

        auto handler = boost::make_shared<pq_handlers::dummy>();
        auto shards = shards_->get();
        auto master = master_from_uid(*shards, uidset);
        if (!master)
        {
            YLOG_CTX_GLOBAL(ctx, error) << "failed to get shard conninfo for uidset " << uidset;
            SAFE_CALL("update_uidset", callback(make_error_code(err_code_storage_fail)));
            return;
        }
        ymod_pq::future_result fres = pq_->request(
            ctx,
            master->conninfo,
            "update_subscriptions_uidset",
            args,
            handler,
            true,
            req_deadline,
            ymod_pq::request_target::master);
        fres.add_callback(boost::bind(
            handle_call_result<decltype(fres), decltype(callback)>, ctx, fres, callback));
    }

    void update_callback(
        task_context_ptr ctx,
        const string& uid,
        const string& service,
        const string& subscription_id,
        const string& old_cb,
        const string& new_cb,
        const update_uidset_callback_t& callback) override
    {
        auto args = boost::make_shared<ymod_pq::bind_array>();
        ymod_pq::push_const_string(args, uid);
        ymod_pq::push_const_string(args, static_cast<const string&>(service));
        ymod_pq::push_const_string(args, subscription_id);
        if (old_cb.size())
        {
            ymod_pq::push_const_string(args, old_cb);
        }
        else
        {
            ymod_pq::push_null(args, ymod_pq::bind_array::STRING);
        }
        ymod_pq::push_const_string(args, new_cb);

        auto handler = boost::make_shared<pq_handlers::dummy>();
        auto shards = shards_->get();
        auto master = master_from_uid(*shards, uid);
        if (!master)
        {
            YLOG_CTX_GLOBAL(ctx, error) << "failed to get shard conninfo for uid " << uid;
            SAFE_CALL("update_callback", callback(make_error_code(err_code_storage_fail)));
            return;
        }
        ymod_pq::future_result fres = pq_->request(
            ctx,
            master->conninfo,
            "update_callback",
            args,
            handler,
            true,
            req_deadline,
            ymod_pq::request_target::master);
        fres.add_callback(boost::bind(
            handle_call_result<decltype(fres), decltype(callback)>, ctx, fres, callback));
    }

    void batch_find(
        task_context_ptr ctx,
        const string& uid,
        const std::vector<string>& services,
        db_role db_role,
        const find_callback_t& callback) override
    {
        auto args = boost::make_shared<ymod_pq::bind_array>();
        ymod_pq::push_const_string(args, uid);
        ymod_pq::push_const_svector(args, services);

        auto handler = boost::make_shared<pq_handlers::list>(user_id(uid));
        auto shards = shards_->get();
        auto master = master_from_uid(*shards, uid);
        if (!master)
        {
            YLOG_CTX_GLOBAL(ctx, error) << "failed to get shard conninfo for uid " << uid;
            SAFE_CALL(
                "service_batch_find", callback(make_error_code(err_code_storage_fail), sub_list()));
            return;
        }
        ymod_pq::future_result fres = batch_pq_->request(
            ctx,
            master->conninfo,
            "batch_list_multiple_services",
            args,
            handler,
            true,
            req_deadline,
            target(db_role));
        fres.add_callback(boost::bind(
            handle_find_result<ymod_pq::future_result, find_callback_t, pq_handlers::list>,
            ctx,
            fres,
            callback,
            handler));
    }

    void enable_fallback(task_context_ptr ctx) override
    {
        auto shards = shards_->get();
        for (auto& shard : *shards)
        {
            pq_->enable_fallback(ctx, shard.master.conninfo, ymod_pq::request_target::master);
            batch_pq_->enable_fallback(ctx, shard.master.conninfo, ymod_pq::request_target::master);
        }
    }

    void disable_fallback(task_context_ptr ctx) override
    {
        auto shards = shards_->get();
        for (auto& shard : *shards)
        {
            pq_->disable_fallback(ctx, shard.master.conninfo, ymod_pq::request_target::master);
            batch_pq_->disable_fallback(
                ctx, shard.master.conninfo, ymod_pq::request_target::master);
        }
    }

private:
    ymod_pq::request_target target(db_role role)
    {
        switch (role)
        {
        case db_role::master:
            return ymod_pq::request_target::try_master;
        case db_role::replica:
            return ymod_pq::request_target::try_replica;
        }
    }
    mod_pq_type pq_;
    mod_pq_type batch_pq_;
    mod_shards_type shards_;
};

}}
