#pragma once

#include "interceptor.h"
#include "query.h"
#include <resharding/controller.h>
#include <xtable/xtable.h>
#include <yxiva/core/resharding/pq_migration_storage.h>
#include <ymod_pq/call.h>
#include <yplatform/module.h>
#include <yplatform/ptree.h>
#include <yplatform/util/tuple_unpack.h>
#include <queue>

namespace yxiva { namespace hub { namespace xtable { namespace resharding {

namespace {
template <typename QueryTraits, typename Callback>
void reply_error(const error_code& ec, Callback&& cb)
{
    yplatform::util::call_with_tuple_args(
        ec_binder<Callback>{ ec, std::forward<Callback>(cb) }, typename QueryTraits::cb_params{});
}

template <typename Params>
inline error_code try_unpack(
    task_context_ptr ctx,
    const string& source,
    response_body<Params>& unpacked_response)
{
    try
    {
        unpack(source, unpacked_response);
        return make_error_code(static_cast<hub::err_code>(unpacked_response.ec_value));
    }
    catch (const std::exception& e)
    {
        YLOG_CTX_GLOBAL(ctx, error) << "failed to unpack redirected query params: " << e.what();
        return make_error_code(err_code_resharding_invalid_callback_params);
    }
}
}

class impl
    : public yplatform::module
    , public hub::resharding::controller
    , public interceptor<impl>
{
    using migration_state = yxiva::resharding::migration::state_type;
    using migrations = yxiva::resharding::pq_migration_storage<ymod_pq::call>;
    using fetch_migrations_callback = std::function<void(const error_code&)>;

    struct migration
    {
        task_context_ptr ctx;
        std::queue<std::function<void()>> queries;
        request_reaction reaction;
        timers::timer_ptr timer;
        timers::timer_ptr query_timer;
    };

public:
    impl(yplatform::reactor& reactor, const yplatform::ptree& config)
    {
        auto xtable_name = config.get<string>("xtable", "direct_xtable");
        xtable_ = yplatform::find<XTable>(xtable_name);

        string migration_storage_name = config.get("migrations", "migrations_xtable");
        migration_storage_ = yplatform::find<migrations, std::shared_ptr>(migration_storage_name);

        http_client_ = yplatform::find<ymod_http_client::call>("http_client");

        max_queue_size_ = config.get<std::size_t>("queue.max_size", 100000);
        queue_timeout_ = config.get<time_duration>("queue.timeout");

        redirect_port_ = config.get<uint16_t>("redirect.port");
        redirect_path_ = config.get<string>("redirect.path");
        redirect_options_.timeouts.total = config.get<time_duration>("redirect.timeout");

        migration_timeout_ = config.get<time_duration>("migration_timeout");
        timers_.reset(new timers::queue(yplatform::reactor::make_not_owning_copy(reactor)));

        using strand = boost::asio::io_service::strand;
        strand_ = make_shared<strand>(*reactor.io());
    }

    virtual std::shared_ptr<shard_config::storage> shards() override
    {
        return xtable_->shards();
    }

    virtual void enable_fallback(task_context_ptr ctx) override
    {
        return xtable_->enable_fallback(ctx);
    }

    virtual void disable_fallback(task_context_ptr ctx) override
    {
        return xtable_->disable_fallback(ctx);
    }

    virtual void force_db_role(std::optional<db_role> r) override
    {
        return xtable_->force_db_role(r);
    }

    virtual void prepare_migration(
        task_context_ptr ctx,
        gid_t gid,
        role expected_role,
        role actual_role,
        const string& master_address,
        const set_state_callback& cb) override
    {
        strand_->dispatch(std::bind(
            &impl::set_migration_state,
            shared_from(this),
            ctx,
            gid,
            expected_role,
            actual_role,
            master_address,
            migration_state::ready,
            cb));
    }

    virtual void abort_migration(
        task_context_ptr ctx,
        gid_t gid,
        role expected_role,
        role actual_role,
        const string& master_address,
        const set_state_callback& cb) override
    {
        strand_->dispatch(std::bind(
            &impl::set_migration_state,
            shared_from(this),
            ctx,
            gid,
            expected_role,
            actual_role,
            master_address,
            migration_state::pending,
            cb));
    }

    virtual void start_migration(
        task_context_ptr ctx,
        gid_t gid,
        role expected_role,
        role actual_role,
        const string& master_address,
        request_reaction reaction,
        const set_state_callback& cb) override
    {
        strand_->dispatch(std::bind(
            &impl::set_migration_state_reaction,
            shared_from(this),
            ctx,
            gid,
            expected_role,
            actual_role,
            master_address,
            migration_state::inprogress,
            reaction,
            cb));
    }

    virtual void finalize_migration(
        task_context_ptr ctx,
        gid_t gid,
        role expected_role,
        role actual_role,
        const string& master_address,
        const set_state_callback& cb) override
    {
        strand_->dispatch(std::bind(
            &impl::set_migration_state,
            shared_from(this),
            ctx,
            gid,
            expected_role,
            actual_role,
            master_address,
            migration_state::finished,
            cb));
    }

    virtual void list_migrations(task_context_ptr ctx, const list_migrations_callback& cb) override
    {
        migration_storage_->fetch_migrations(
            [ctx, cb, migration_storage = migration_storage_](const operation::result& r) {
                if (!r)
                {
                    YLOG_CTX_GLOBAL(ctx, error)
                        << "failed to fetch resharding migrations from storage " << r.error_reason;
                    cb(make_error_code(err_code_migration_storage_fail), {});
                }
                else
                {
                    cb(error_code{}, *migration_storage->get());
                }
            });
    }

    template <typename Method, typename... Args>
    void execute(Method method, Args&&... args)
    {
        (xtable_.get()->*method)(std::forward<Args>(args)...);
    }

    template <typename QueryTraits, typename Callback, typename... Args>
    void intercept(
        task_context_ptr ctx,
        QueryTraits /*tag*/,
        gid_t gid,
        Callback&& cb,
        Args&&... args)
    {
        auto state = get_migration_state(gid);
        if (!state)
        {
            yplatform::util::call_with_tuple_args(
                ec_binder<Callback>{ make_error_code(err_code_storage_fail),
                                     std::forward<Callback>(cb) },
                typename QueryTraits::cb_params{});
            return;
        }
        switch (*state)
        {
        case migration_state::pending:
        case migration_state::finished:
            execute(
                QueryTraits::method(),
                ctx,
                std::forward<Args>(args)...,
                std::forward<Callback>(cb));
            break;

        case migration_state::ready:
        case migration_state::inprogress:
            redirect_or_enqueue<QueryTraits>(
                ctx,
                gid,
                state == migration_state::inprogress,
                std::forward<Callback>(cb),
                std::forward<Args>(args)...);
            break;
        }
    }

private:
    template <typename QueryTraits, typename Callback, typename... Args>
    void redirect_or_enqueue(
        task_context_ptr ctx,
        gid_t gid,
        bool enqueue_when_master,
        Callback&& cb,
        Args&&... args)
    {
        switch (role_)
        {
        case role::unknown:
            reply_error<QueryTraits>(
                make_error_code(err_code_storage_fail), std::forward<Callback>(cb));
            break;
        case role::master:
            if (enqueue_when_master)
            {
                auto cb_copy = std::remove_reference_t<Callback>{ cb };
                enqueue<QueryTraits>(
                    ctx,
                    gid,
                    std::bind(
                        QueryTraits::method(), xtable_, ctx, std::forward<Args>(args)..., cb_copy),
                    cb_copy);
            }
            else
            {
                execute(
                    QueryTraits::method(),
                    ctx,
                    std::forward<Args>(args)...,
                    std::forward<Callback>(cb));
            }
            break;
        case role::slave:
            redirect<QueryTraits>(ctx, std::forward<Callback>(cb), std::forward<Args>(args)...);
            break;
        }
    }

    template <typename QueryTraits, typename Callback, typename... Args>
    void redirect(task_context_ptr ctx, Callback&& cb, Args&&... args)
    {
        namespace http = ymod_httpclient;

        string url = make_redirect_url(QueryTraits::get_type());
        auto params = typename QueryTraits::params{ std::forward<Args>(args)... };
        auto request = http::request::POST(url, pack(params));
        http_client_->async_run(
            ctx,
            request,
            redirect_options_,
            [ctx, cb = std::move(cb)](const error_code& ec, const http::response& response) {
                response_body<typename QueryTraits::cb_params> unpacked_response;
                error_code method_ec;
                if (ec)
                {
                    method_ec = ec;
                }
                else if (response.status != 200)
                {
                    method_ec = make_error_code(err_code_remote_xtable_fail);
                }
                else
                {
                    method_ec = try_unpack(ctx, response.body, unpacked_response);
                }
                if (method_ec)
                {
                    YLOG_CTX_GLOBAL(ctx, error) << "failed to redirect query "
                                                << query_type_to_string(QueryTraits::get_type())
                                                << " error: " << method_ec.message();
                }
                yplatform::util::call_with_tuple_args(
                    ec_binder<Callback>{ method_ec, cb }, unpacked_response.params);
            });
    }

    template <typename QueryTraits, typename BoundQuery, typename Callback>
    void enqueue(task_context_ptr ctx, gid_t gid, BoundQuery&& bound_query, Callback&& cb)
    {
        strand_->dispatch([this,
                           self = shared_from(this),
                           ctx,
                           gid,
                           cb = std::move(cb),
                           q = std::move(bound_query)]() {
            auto it = migrations_.find(gid);
            if (it == migrations_.end())
            {
                reply_error<QueryTraits>(make_error_code(err_code_resharding_queue_not_found), cb);
                return;
            }

            auto& migration = it->second;
            if (migration.queries.size() == max_queue_size_)
            {
                reply_error<QueryTraits>(make_error_code(err_code_resharding_queue_overflow), cb);
                return;
            }

            if (migration.reaction == request_reaction::reject)
            {
                reply_error<QueryTraits>(make_error_code(err_code_resharding_queue_reject), cb);
                return;
            }

            if (!migration.query_timer)
            {
                migration.query_timer = timers_->create_timer();
                migration.query_timer->async_wait(
                    queue_timeout_,
                    strand_->wrap(
                        [gid, ctx = migration.ctx, this, self = yplatform::shared_from(this)]() {
                            handle_migration_timeout(ctx, gid);
                        }));
            }

            migration.queries.push(std::move(q));
        });
    }

    string make_redirect_url(query_type t)
    {
        return redirect_url_ + query_type_to_string(t);
    }

    void update_redirect_url(const string& master_address)
    {
        redirect_url_ =
            master_address + ":" + std::to_string(redirect_port_) + redirect_path_ + "?type=";
    }

    void set_migration_state(
        task_context_ptr ctx,
        gid_t gid,
        role expected_role,
        role actual_role,
        const string& master_address,
        migration_state gid_state,
        const set_state_callback& cb)
    {
        set_migration_state_reaction(
            ctx, gid, expected_role, actual_role, master_address, gid_state, {}, cb);
    }

    void set_migration_state_reaction(
        task_context_ptr ctx,
        gid_t gid,
        role expected_role,
        role actual_role,
        const string& master_address,
        migration_state gid_state,
        request_reaction queue_reaction,
        const set_state_callback& cb)
    {
        role_ = actual_role;
        if (expected_role != actual_role)
        {
            cb(make_error_code(err_code_resharding_role_mismatch), {});
            return;
        }
        switch (actual_role)
        {
        case role::unknown:
            cb(make_error_code(err_code_resharding_role_mismatch), {});
            break;
        case role::master:
            write_migration_storage(ctx, gid, gid_state, queue_reaction, cb);
            break;
        case role::slave:
            update_redirect_url(master_address);
            fetch_migrations(ctx, std::bind(cb, std::placeholders::_1, migration_state::pending));
            break;
        }
    }

    void write_migration_storage(
        task_context_ptr ctx,
        gid_t gid,
        migration_state gid_state,
        request_reaction queue_reaction,
        const set_state_callback& cb)
    {
        migration_storage_->set_migration_state(
            gid,
            gid_state,
            strand_->wrap(
                [this,
                 self = yplatform::shared_from(this),
                 ctx,
                 gid,
                 gid_state,
                 queue_reaction,
                 cb](const operation::result& r, migration_state new_state, bool bad_state) {
                    if (!r)
                    {
                        YLOG_CTX_GLOBAL(ctx, error)
                            << "failed to set migration state for gid " << gid << " to "
                            << to_string(gid_state) << ", because: " << r.error_reason;
                        cb(make_error_code(err_code_migration_storage_fail), new_state);
                    }
                    else if (bad_state)
                    {
                        YLOG_CTX_GLOBAL(ctx, error)
                            << "failed to set migration state for gid " << gid << " to "
                            << to_string(gid_state) << ", because it is in unexpected state "
                            << to_string(new_state);
                        cb(make_error_code(err_code_resharding_bad_migration_state), new_state);
                    }
                    else
                    {
                        cb(update_migration(ctx, gid, gid_state, queue_reaction), new_state);
                    }
                }));
    }

    void fetch_migrations(task_context_ptr ctx, const fetch_migrations_callback& cb)
    {
        migration_storage_->fetch_migrations(
            [ctx, cb, migration_storage = migration_storage_](const operation::result& r) {
                if (!r)
                {
                    YLOG_CTX_GLOBAL(ctx, error)
                        << "failed to fetch resharding migrations from storage " << r.error_reason;
                    cb(make_error_code(err_code_migration_storage_fail));
                }
                else
                {
                    cb(error_code{});
                }
            });
    }

    // Executed in strand.
    error_code update_migration(
        task_context_ptr ctx,
        gid_t gid,
        migration_state gid_state,
        request_reaction reaction)
    {
        switch (gid_state)
        {
        case migration_state::pending:
            return abort_migration(gid);
        case migration_state::finished:
            return finalize_migration(gid);
        case migration_state::ready:
            return {};
        case migration_state::inprogress:
            return start_migration(ctx, gid, reaction);
        }
        return {};
    }

    // Executed in strand.
    error_code start_migration(task_context_ptr ctx, gid_t gid, request_reaction reaction)
    {
        auto it = migrations_.find(gid);
        if (it != migrations_.end())
        {
            return make_error_code(err_code_resharding_queue_already_exists);
        }

        auto& migration = migrations_[gid];
        // Preserve start_migration context id for logging.
        migration.ctx = yxiva::make_shared<yplatform::task_context>(ctx->uniq_id());
        migration.reaction = reaction;
        migration.timer = timers_->create_timer();
        migration.timer->async_wait(
            migration_timeout_,
            strand_->wrap([gid, ctx = migration.ctx, this, self = yplatform::shared_from(this)]() {
                handle_migration_timeout(ctx, gid);
            }));
        return {};
    }

    // Executed in strand.
    error_code abort_migration(gid_t gid)
    {
        // Queue is not required to exist for aborting.
        finalize_migration(gid);
        return {};
    }

    // Executed in strand.
    error_code finalize_migration(gid_t gid)
    {
        auto it = migrations_.find(gid);
        if (it == migrations_.end())
        {
            return make_error_code(err_code_resharding_queue_not_found);
        }

        auto& queries = it->second.queries;
        while (!queries.empty())
        {
            auto& q = queries.front();
            q();
            queries.pop();
        }

        migrations_.erase(it); // also cancels timers
        return {};
    }

    // Executed in strand.
    void handle_migration_timeout(task_context_ptr ctx, gid_t gid)
    {
        static const string abort_reason = "migration timed out";
        log_and_abort_migration(ctx, gid, abort_reason);
    }

    void abort_my_migrations(const std::string& reason)
    {
        for (auto& p : migrations_)
        {
            auto gid = p.first;
            auto ctx = p.second.ctx;

            log_and_abort_migration(ctx, gid, reason);
        }
    }

    void log_and_abort_migration(task_context_ptr ctx, gid_t gid, const string& reason)
    {
        YLOG_CTX_GLOBAL(ctx, error) << "aborting migration of gid " << gid << ": " << reason;
        abort_migration(
            ctx,
            gid,
            role::master,
            role::master,
            "",
            [ctx, gid](const error_code& ec, migration_state) {
                if (ec)
                {
                    YLOG_CTX_GLOBAL(ctx, error)
                        << "aborting migration of gid " << gid << " failed: " << ec.message();
                }
                else
                {
                    YLOG_CTX_GLOBAL(ctx, info) << "aborted migration of gid " << gid;
                }
            });
    }

    boost::optional<migration_state> get_migration_state(gid_t gid)
    {
        auto migrations = migration_storage_->get();
        auto it = std::lower_bound(
            migrations->begin(),
            migrations->end(),
            yxiva::resharding::migration{ migration_state{}, 0, gid });

        return it != migrations->end() ? it->state : boost::optional<migration_state>{};
    }

    shared_ptr<XTable> xtable_;
    shared_ptr<ymod_httpclient::call> http_client_;
    std::shared_ptr<migrations> migration_storage_;

    std::atomic<role> role_;
    string redirect_url_;

    std::size_t max_queue_size_;
    time_duration queue_timeout_;
    uint16_t redirect_port_;
    string redirect_path_;
    ymod_httpclient::options redirect_options_;

    time_duration migration_timeout_;
    timers::queue_ptr timers_;

    shared_ptr<boost::asio::io_service::strand> strand_;
    std::unordered_map<gid_t, migration> migrations_;
};

}}}}