#include <ymod_messenger/module.h>
#include <ymod_lease/node.h>
#include <iostream>

auto messenger_conf()
{
    return yplatform::ptree();
}

auto lease_conf()
{
    yplatform::ptree lease_conf;
    lease_conf.put("arbiters_count", 5);
    lease_conf.put("tag", "testtag");
    lease_conf.put("lease_log_id", "global");
    lease_conf.put("verbose_logging", "1");
    return lease_conf;
}

int main(int argc, char* argv[])
{
    auto usage = [&argv](){
        std::cout << "usage:\n"
            << argv[0] << " read <resource> <arbiters-list>\n"
            << argv[0] << " write <resource> <value> <arbiters-list>\n";
    };

    if (argc <= 4) {
        usage();
        return 1;
    }

    std::string command = argv[1];
    std::string target_value;
    int offset = 3;

    if (command == "write") {
        target_value = argv[3];
        offset = 4;
    } else if (command != "read") {
        usage();
        return 1;
    }

    std::string resource = argv[2];
    std::set<std::string> hosts;
    for (int i = offset; i < argc; ++i) {
        hosts.insert(argv[i]);
    }

    boost::asio::io_service io;

    yplatform::log::init_file(io, "log.log");

    auto messenger = std::make_shared<ymod_messenger::module>(io, messenger_conf());
    yplatform::register_module(io, "netch", messenger);

    auto lease_node = std::make_shared<ylease::node>(io, lease_conf());

    messenger->connect_to_cluster(hosts);
    messenger->bind_events([](auto&& addr, auto&& event) {
        L_(info) << "events addr=" << addr << " event=" << event.event
            << " pool=" << event.pool;
    });

    std::string quorum_value;
    std::string resource_owner;

    lease_node->bind(resource,
        [&lease_node, &target_value, &quorum_value, &resource_owner]
        (auto&& resource, auto&& node, auto&& ballot, auto&& value) {
            L_(info) << "BUSY resource=" << resource << " node=" << node
                << " ballot=" << ballot << " value=" << value;
            quorum_value = value;
            resource_owner = node;
            if (node == lease_node->node_id() && target_value != "") {
                lease_node->update_acquire_value(resource, target_value);
            }
        },
        [](auto& resource) {
            L_(info) << "FREE resource=" << resource;
        });

    lease_node->start_acquire_lease(resource);

    io.run_for(std::chrono::seconds(3));

    if (command == "read") {
        std::cout << quorum_value << std::endl;
        return 0;
    }

    if (target_value != quorum_value) {
        std::cerr << "write failed, quorum_value=" << quorum_value
            << " owner=" << resource_owner << std::endl;
    }

    return target_value == quorum_value ? 0 : 1;
}
