#pragma once

#include <mailpusher/types.h>
#include <ymod_ratecontroller/errors.h>
#include <ymod_httpclient/errors.h>

namespace yxiva::mailpusher::error {

enum code
{
    success,
    bad_gateway,        // TODO something more verbose
    remote_bad_request, // Must be logged separately to debug bad request cause.
    internal_error,     // TODO something more verbose
    network_error,
    task_cancelled,
    rate_limit,
    uninitialized_user,
    payload_too_large,
    invalid_subscription_extra,

    COUNT
};

}

namespace boost::system {

template <>
struct is_error_code_enum<yxiva::mailpusher::error::code>
{
    static const bool value = true;
};

}

namespace yxiva::mailpusher::error {

class error_category : public boost::system::error_category
{
public:
    const char* name() const noexcept(true)
    {
        return "mailpusher";
    }

    std::string message(int ev) const
    {
        return error_names.at(ev);
    }

    static const error_category& instance()
    {
        static constexpr std::initializer_list<const char*> names = {
            "success",
            "bad gateway",
            "remote server answer bad request",
            "internal error",
            "network error",
            "task cancelled",
            "rate limit",
            "user not inited",
            "payload too large",
            "invalid subscription extra",
        };
        static_assert(
            names.size() == code::COUNT,
            "Error codes count doesn't correspond with error names count");
        static const error_category category(names);
        return category;
    }

private:
    error_category(std::initializer_list<const char*> names)
        : error_names(names.begin(), names.end())
    {
    }

    const std::vector<std::string> error_names;
};

inline error_code make_error(code ec = error::success)
{
    return error_code(ec, error_category::instance());
}

inline error_code from_rate_controller_error(const error_code& ec)
{
    using ymod_ratecontroller::error;
    switch (static_cast<error>(ec.value()))
    {
    case error::ok:
        return make_error(success);
    case error::capacity_exceeded:
        return make_error(rate_limit);
    case error::task_aborted:
        return make_error(task_cancelled);
    default:
        return make_error(internal_error);
    }
}

inline error_code from_http_response(const error_code& ec, const yhttp::response resp)
{
    if (ec.value() == ymod_httpclient::http_error::task_canceled ||
        ec.value() == ymod_httpclient::http_error::request_timeout)
    {
        return make_error(task_cancelled);
    }
    else if (
        ec.value() == ymod_httpclient::http_error::invalid_url ||
        ec.value() == ymod_httpclient::http_error::unsupported_scheme ||
        ec.value() == ymod_httpclient::http_error::no_service_ticket)
    {
        return make_error(internal_error);
    }
    else if (ec)
    {
        return make_error(network_error);
    }
    else if (resp.status == 429)
    {
        // TODO Different error for external rate limit?
        return make_error(rate_limit);
    }
    else if (resp.status / 100 == 4 && resp.status != 429)
    {
        return make_error(remote_bad_request);
    }
    else if (resp.status / 100 != 2)
    {
        return make_error(bad_gateway);
    }
    return make_error(success);
}

}
