#pragma once

#include <yxiva/core/types.h>
#include <yxiva/core/json.h>
#include <yxiva/core/platforms.h>
#include <variant>

namespace yxiva::mailpusher {
// Xiva subscription view through public Xiva API.
struct subscription
{
    struct common
    {
        string id;
        string client;
        string session;
        string filter;
        ttl_t ttl = 0;
        string extra;
        time_t init_time = 0;
        time_t last_sent = 0;
        local_id_t pos = 0;
    };

    struct mobile : public common
    {
        string app;
        string uuid;
        string platform;
        string device;
    };

    struct http : public common
    {
        string url;
    };

    struct webpush : public common
    {
    };

    subscription() : impl_()
    {
    }
    explicit subscription(const struct mobile& mobile) : impl_(mobile)
    {
    }
    explicit subscription(const struct http& http) : impl_(http)
    {
    }
    explicit subscription(const struct webpush& webpush) : impl_(webpush)
    {
    }

    const auto* common() const
    {
        return std::visit([](auto& arg) { return static_cast<const struct common*>(&arg); }, impl_);
    }
    const auto* mobile() const
    {
        return std::get_if<struct mobile>(&impl_);
    }
    const auto* http() const
    {
        return std::get_if<struct http>(&impl_);
    }
    const auto* webpush() const
    {
        return std::get_if<struct webpush>(&impl_);
    }

    string transport() const
    {
        if (mobile()) return "mobile";
        if (http()) return "http";
        if (webpush()) return "webpush";
        return "unknown";
    }

private:
    std::variant<struct mobile, struct http, struct webpush> impl_;
};

inline void decode_common(const json_value& json_sub, struct subscription::common& common)
{
    common.id = json_get(json_sub, "id", common.id);
    common.client = json_get(json_sub, "client", common.client);
    common.session = json_get(json_sub, "session", common.session);
    common.filter = json_get(json_sub, "filter", common.filter);
    common.ttl = json_get(json_sub, "ttl", common.ttl);
    common.extra = json_get(json_sub, "extra", common.extra);
    common.init_time = json_get(json_sub, "init_time", common.init_time);
    common.last_sent = json_get(json_sub, "last_sent", common.last_sent);
    common.pos = json_get(json_sub, "pos", common.pos);
}

inline auto decode_http(const json_value& json_sub)
{
    struct subscription::http http;
    decode_common(json_sub, http);
    http.url = json_get(json_sub, "url", http.url);
    return http;
}

inline auto decode_mobile(const json_value& json_sub)
{
    struct subscription::mobile mobile;
    decode_common(json_sub, mobile);
    mobile.platform = platform::resolve_alias(json_get(json_sub, "platform", mobile.platform)).name;
    mobile.app = json_get(json_sub, "app", mobile.app);
    mobile.uuid = json_get(json_sub, "uuid", mobile.uuid);
    mobile.device = json_get(json_sub, "device", mobile.device);
    return mobile;
}

inline auto decode_webpush(const json_value& json_sub)
{
    struct subscription::webpush webpush;
    decode_common(json_sub, webpush);
    return webpush;
}

inline std::tuple<operation::result, subscription> decode_subscription(const json_value& json_sub)
{
    if (!json_sub.is_object() || !json_sub.has_member("client"))
    {
        return { "invalid subscription json", {} };
    }
    if (json_sub.has_member("url"))
    {
        return { operation::success, subscription(decode_http(json_sub)) };
    }
    if (json_sub.has_member("platform"))
    {
        return { operation::success, subscription(decode_mobile(json_sub)) };
    }
    return { operation::success, subscription(decode_webpush(json_sub)) };
}

}