#pragma once

#include <mailpusher/types.h>
#include <mailpusher/task.h>
#include <yplatform/module.h>
#include <yplatform/log/typed.h>
#include <optional>

#define TYPED                                                                                      \
    if (logger_) YLOG((*logger_), info)

namespace yxiva::mailpusher {

namespace ll = yplatform::log::typed;

class mod_log : public yplatform::module
{
public:
    mod_log() = default; // For unit test

    mod_log(const yplatform::ptree& conf)
        : logger_(
              yplatform::log::tskv_logger{ YGLOBAL_LOG_SERVICE, conf.get("typed_log_id", "tskv") })
    {
    }

    void sent(const shared_ptr<task>& task, const event& event)
    {
        TYPED << ll::make_attr("status", "sent") << make_attrs(*task) << make_attrs(event);
    }

    void dropped(const shared_ptr<task>& task, const event& event, const string& reason)
    {
        TYPED << ll::make_attr("status", "dropped") << ll::make_attr("reason", reason)
              << make_attrs(*task) << make_attrs(event);
    }

    void failed(const shared_ptr<task>& task, const string& reason)
    {
        TYPED << ll::make_attr("status", "failed") << ll::make_attr("reason", reason)
              << make_attrs(*task);
    }

    void failed(const shared_ptr<task>& task, const event& event, const string& reason)
    {
        TYPED << ll::make_attr("status", "failed") << ll::make_attr("reason", reason)
              << make_attrs(*task) << make_attrs(event);
    }

    ll::attributes_map make_attrs(const task& task)
    {
        using ymod_webserver::prof_time_to_string;
        ll::attributes_map am;
        am << ll::make_attr("context", task.uniq_id()) << ll::make_attr("uid", task.uid)
           << ll::make_attr("suid", task.suid) << ll::make_attr("dbname", task.db)
           << ll::make_attr("counters_fetched", get_status_name(task.counters_fetched));
        for (auto& prof : task.profilers.get_values())
        {
            am << ll::make_attr("profiler_" + prof.name, prof_time_to_string(prof.time));
        }
        am << ll::make_attr("profiler_total", prof_time_to_string(task.profilers.get_total()));
        return am;
    }

    ll::attributes_map make_attrs(const event& event)
    {
        ll::attributes_map am;
        am << ll::make_attr("action", get_action_name(event.action_type))
           << ll::make_attr("cid", event.change_id) << ll::make_attr("ts", std::to_string(event.ts))
           << ll::make_attr("lag", std::to_string(event.lag()))
           << ll::make_attr("item_count", std::to_string(event.items.size()))
           << ll::make_attr("meta_count", std::to_string(event.metadata_count))
           << ll::make_attr("avatar_fetched", get_status_name(event.avatar_fetched));
        if (event.xiva_context.size())
        {
            am << ll::make_attr("xiva_context", event.xiva_context);
        }
        if (event.xiva_transit.size())
        {
            am << ll::make_attr("transit_id", event.xiva_transit);
        }
        return am;
    }

private:
    std::optional<yplatform::log::tskv_logger> logger_;
};

}
