#pragma once

#include "metadata_fetcher.h"
#include "settings.h"
#include <mailpusher/task.h>
#include <mailpusher/types.h>
#include <mailpusher/errors.h>
#include <ymod_ratecontroller/rate_controller.h>
#include <yplatform/coroutine.h>
#include <yplatform/util/safe_call.h>
#include <boost/range/adaptors.hpp>
#include <map>
#include <set>
#include <vector>

#include "processor_def.h"

namespace yxiva::mailpusher {

namespace hacks {

inline bool avatars_enabled(const string& uid, unsigned disabled_percent)
{
    return std::strtoull(uid.c_str(), nullptr, 10) % 100 >= disabled_percent;
}

}

template <typename HttpClient>
struct avatar_fetcher
{
    using metadata_fetcher = metadata_fetcher<HttpClient>;
    using yield_context_t = yplatform::yield_context<avatar_fetcher<HttpClient>>;
    struct job
    {
        job(event& event, const string& email) : event(event), email(email)
        {
        }

        event& event;
        string email;
    };

    shared_ptr<task> task;
    ymod_ratecontroller::rate_controller_ptr rate_controller;
    HttpClient& http_client;
    const settings& settings;
    callback_t cb;

    ymod_ratecontroller::completion_handler rate_controller_on_complete{};
    error_code error{};
    json_value avatars{};

    std::vector<job> jobs = {};

    static const string& name()
    {
        static const string NAME = "ava";
        return NAME;
    }

    void operator()(
        yield_context_t yield_context,
        const error_code& ec = {},
        yhttp::response resp = {})
    {
        try
        {
            reenter(yield_context)
            {
                yield rate_controller->post(yield_context, "", task->deadline());
                if (error = error::from_rate_controller_error(ec))
                {
                    yield break;
                }
                if (prepare_jobs())
                {
                    yield do_request(yield_context);
                    if (error = error::from_http_response(ec, resp))
                    {
                        if (ec)
                        {
                            LOG_ERROR("http_error", ec.message());
                        }
                        else
                        {
                            LOG_ERROR("http_response", "code " + std::to_string(resp.status));
                        }
                    }
                    else
                    {
                        error = process_response(resp);
                    }
                }
                if (error)
                {
                    fail_all();
                }
                else
                {
                    fill_events();
                }
            }
        }
        catch (const std::exception& ex)
        {
            LOG_ERROR("exception", ex.what());
            error = make_error(error::internal_error);
        }

        if (yield_context.is_complete())
        {
            cb(error);
            yplatform::safe_call(rate_controller_on_complete);
        }
    }

    string make_emails_param()
    {
        using boost::adaptors::transformed;
        static const auto email_getter = [](auto& job) { return job.email; };
        return boost::algorithm::join(jobs | transformed(email_getter), "&email=");
    }

    void do_request(yield_context_t yield_context)
    {
        http_client.async_run(
            task,
            yhttp::request::POST(
                settings.ava.request,
                yhttp::form_encode({ { "json", "1" }, { "email", "" } }) + make_emails_param()),
            yield_context);
    }

    // Rate controller handler.
    void operator()(
        yield_context_t yield_context,
        const error_code& ec,
        ymod_ratecontroller::completion_handler on_complete)
    {
        rate_controller_on_complete = std::move(on_complete);
        (*this)(yield_context, ec);
    }

    error_code process_response(const yhttp::response& resp)
    {
        // Handle retry by equalizer?
        if (auto res = json_parse(avatars, resp.body); !res)
        {
            LOG_ERROR("malformed_response", res.error_reason);
            return make_error(error::bad_gateway);
        }
        return make_error(error::success);
    }

    string extract_sender(const event& event)
    {
        if (event.items.size())
        {
            auto& item = event.items.front();
            if (item.has_member("from") && item["from"].is_array() && item["from"].size() == 1)
            {
                auto& from = item["from"][0UL];
                auto email = json_get<string>(from, "local", "") + "@" +
                    json_get<string>(from, "domain", "");
                return email;
            }
        }
        return {};
    }

    size_t prepare_jobs()
    {
        size_t count = 0;
        for (auto i = task->processed;
             i < task->events.size() && count < settings.ava.max_emails_in_profiles_request;
             ++i)
        {
            auto& event = task->events[i];
            if (!required(event, settings)) continue;
            if (auto email = extract_sender(event); email.size())
            {
                jobs.emplace_back(event, email);
                ++count;
            }
            else
            {
                event.avatar_fetched = fetch_status::SKIPPED;
            }
        }
        return count;
    }

    void fill_events()
    {
        for (auto& job : jobs)
        {
            if (auto avatar = json_get<string>(avatars, job.email.data(), ""); avatar.size())
            {
                job.event.avatar_fetched = fetch_status::FETCHED;
                for (auto& item : job.event.items)
                {
                    item["avatarUrl"] = avatar;
                }
            }
            else
            {
                // Failed request or missing result.
                job.event.avatar_fetched = fetch_status::MISSING;
            }
        }
    }

    void fail_all()
    {
        for (auto& job : jobs)
        {
            job.event.avatar_fetched = fetch_status::FAILED;
        }
    }

    static bool required(const event& ev, const struct settings& settings)
    {
        return hacks::avatars_enabled(ev.uid, settings.ava.disabled_percent) &&
            ev.action_type == action::NEW_MAIL && ev.avatar_fetched == fetch_status::NONE &&
            metadata_fetcher::done(ev);
    }
};

}

#include "processor_undef.h"