#pragma once

#include "counters_fetcher_impl.h"
#include "settings.h"
#include "../mod_log/mod_log.h"
#include <mailpusher/task.h>
#include <mailpusher/types.h>
#include <mailpusher/errors.h>
#include <yplatform/coroutine.h>
#include <yplatform/util/safe_call.h>

#include "processor_def.h"

namespace yxiva::mailpusher {

template <typename HttpClient>
struct counters_fetcher
{
    using yield_context_t = yplatform::yield_context<counters_fetcher>;
    using impl_t = counters_fetcher_impl<HttpClient>;

    shared_ptr<task> task;
    rate_controllers<ymod_ratecontroller::rate_controller_ptr> rate_controllers;
    http_clients<HttpClient&> http_clients;
    const settings& settings;
    callback_t cb;

    std::unordered_map<string, device>::iterator device_it = {};
    std::size_t account_index = 0;

    static const string& name()
    {
        return impl_t::name();
    }

    void operator()(
        yield_context_t yield_context,
        const error_code& ec = {},
        const typename impl_t::counters_t& counters = {})
    {
        reenter(yield_context)
        {
            yield fetch_counters(task->uid, settings.environment, yield_context);
            task->counters_fetched = ec ? fetch_status::FAILED : fetch_status::FETCHED;
            task->counters = counters;
            if (!ec)
            {
                auto [json_counters, json_counters_new] = compose_json_counters(counters);
                for (auto& event : task->events)
                {
                    if (!required(event)) continue;
                    event.args["counters"] = json_counters;
                    event.args["countersNew"] = json_counters_new;
                }
            }
            device_it = task->devices.begin();
            for (; device_it != task->devices.end(); ++device_it)
            {
                account_index = 0;
                for (; account_index < device_it->second.accounts.size(); ++account_index)
                {
                    if (counters_required(next_unprocessed_account()))
                    {
                        yield fetch_counters(next_unprocessed_account(), yield_context);
                        if (!ec)
                        {
                            next_unprocessed_account().counters_fetched = fetch_status::FETCHED;
                            next_unprocessed_account().fids_counters = counters;
                        }
                    }
                }
            }
            cb(ec);
        }
    }

    void operator()(typename yield_context_t::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& e)
        {
            LOG_ERROR("exception", e.what());
        }
        yplatform::safe_call(cb, make_error(error::internal_error));
    }

    std::tuple<json_value, json_value> compose_json_counters(
        const typename impl_t::counters_t& counters)
    {
        auto json_counters = json_value(json_type::tarray);
        auto json_counters_new = json_value(json_type::tarray);
        for (auto& c : counters)
        {
            if (c.fid && c.count)
            {
                json_counters.push_back(c.fid);
                json_counters.push_back(c.count);
            }
            if (c.fid && c.count_new)
            {
                json_counters_new.push_back(c.fid);
                json_counters_new.push_back(c.count_new);
            }
        }
        return { json_counters, json_counters_new };
    }

    static bool required(const event& e)
    {
        return e.action_type == action::NEW_MAIL;
    }

private:
    bool counters_required(const account& account)
    {
        return account.has_subscriptions && account.badge_enabled;
    }

    void fetch_counters(const account& account, yield_context_t yield_context)
    {
        if (account.uid == task->uid && account.environment == settings.environment)
        {
            operator()(yield_context, {}, task->counters);
        }
        else
        {
            fetch_counters(account.uid, account.environment, yield_context);
        }
    }

    void fetch_counters(const string& uid, const string& environment, yield_context_t yield_context)
    {
        yplatform::spawn(impl_t{ task,
                                 uid,
                                 rate_controllers.at(environment),
                                 http_clients.at(environment),
                                 settings.counters.at(environment),
                                 yield_context });
    }

    auto& next_unprocessed_account()
    {
        return device_it->second.accounts.at(account_index);
    }
};

}

#include "processor_undef.h"