#pragma once

#include "settings.h"
#include "../mod_log/mod_log.h"
#include <mailpusher/fid_counters.h>
#include <mailpusher/types.h>
#include <mailpusher/errors.h>
#include <ymod_ratecontroller/rate_controller.h>
#include <yplatform/coroutine.h>
#include <yplatform/util/safe_call.h>

#include "processor_def.h"

namespace yxiva::mailpusher {

template <typename HttpClient>
struct counters_fetcher_impl
{
    using yield_context_t = yplatform::yield_context<counters_fetcher_impl>;
    using counters_t = std::vector<fid_counters>;
    using counters_callback_t = std::function<void(const error_code&, const counters_t&)>;

    yplatform::task_context_ptr ctx;
    string uid;
    ymod_ratecontroller::rate_controller_ptr rate_controller;
    HttpClient& http_client;
    const struct settings::counters& settings;
    counters_callback_t cb;

    ymod_ratecontroller::completion_handler rate_controller_on_complete{};
    error_code error{};
    counters_t counters{};

    static const string& name()
    {
        static const string NAME = "counters";
        return NAME;
    }

    void operator()(
        yield_context_t yield_context,
        const error_code& ec = {},
        yhttp::response resp = {})
    {
        reenter(yield_context)
        {
            yield rate_controller->post(yield_context, "", ctx->deadline());
            if (error = error::from_rate_controller_error(ec))
            {
                yield break;
            }
            yield do_request(yield_context);
            if (error = error::from_http_response(ec, resp))
            {
                if (ec)
                {
                    LOG_ERROR_CTX(ctx, "http_error", ec.message());
                }
                else
                {
                    LOG_ERROR_CTX(ctx, "http_response", "code " + std::to_string(resp.status));
                }
                yield break;
            }
            error = process_response(resp);
        }

        if (yield_context.is_complete())
        {
            cb(error, counters);
            rate_controller_on_complete();
        }
    }

    void operator()(typename yield_context_t::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& e)
        {
            LOG_ERROR_CTX(ctx, "exception", e.what());
        }
        yplatform::safe_call(cb, make_error(error::internal_error), counters);
        yplatform::safe_call(rate_controller_on_complete);
    }

    void do_request(yield_context_t yield_context)
    {
        http_client.async_run(
            ctx,
            yhttp::request::GET(settings.request + yhttp::url_encode({ { "uid", uid } })),
            yield_context);
    }

    // Rate controller handler.
    void operator()(
        yield_context_t yield_context,
        const error_code& ec,
        ymod_ratecontroller::completion_handler on_complete)
    {
        rate_controller_on_complete = std::move(on_complete);
        (*this)(yield_context, ec);
    }

    error_code process_response(const yhttp::response& resp)
    {
        json_value json_resp;
        if (auto res = json_parse(json_resp, resp.body); !res)
        {
            LOG_ERROR_CTX(ctx, "malformed_response", res.error_reason);
            return make_error(error::bad_gateway);
        }
        if (json_resp.has_member("error"))
        {
            LOG_ERROR_CTX(ctx, "bad_response", json_write(json_resp["error"]));
            return make_error(error::bad_gateway);
        }
        if (!decode_counters(json_resp))
        {
            return make_error(error::bad_gateway);
        }
        return make_error(error::success);
    }

    bool decode_counters(const json_value& json_counters)
    {
        if (!json_counters.has_member("folders") || !json_counters["folders"].is_object())
        {
            return false;
        }
        auto&& folders = json_counters["folders"];
        for (auto it = folders.members_begin(); it != folders.members_end(); ++it)
        {
            auto fid = std::stoul(string(it.key()));
            counters.push_back(
                fid_counters{ fid, json_get(*it, "cnt", 0U), json_get(*it, "new", 0U) });
        }
        return true;
    }
};

}

#include "processor_undef.h"