#pragma once

#include <mailpusher/task.h>
#include <mailpusher/event.h>
#include <mailpusher/action.h>
#include <yxiva/core/json.h>
#include <map>
#include <vector>
#include <optional>

namespace yxiva::mailpusher {

struct field
{
    struct context
    {
        const event& event;
        const subscription& subscription;
    };

    typedef std::function<json_value(const json_value_ref&)> transformation_t;
    typedef std::function<json_value(const context&)> context_getter_t;
    typedef std::function<bool(const context&)> context_matcher_t;

    string target;
    string source;
    json_value default_value = "";

    bool is_optional = false;
    bool from_args = false;
    bool is_array = false;
    bool force_string = false;
    bool force_array_to_string = false;
    transformation_t transformation;
    context_getter_t get_from_context;
    context_matcher_t context_matcher;
    std::vector<field> children;

    field(const string& target_) : target(target_)
    {
    }

    bool is_composite() const
    {
        return children.size() > 0;
    }
    bool matches_context(const context& ctx) const
    {
        return !context_matcher || context_matcher(ctx);
    }

    field& from_item(const string& s)
    {
        source = s;
        from_args = false;
        return *this;
    }
    field& from_arg(const string& s)
    {
        source = s;
        from_args = true;
        return *this;
    }
    field& optional()
    {
        is_optional = true;
        return *this;
    }
    field& array()
    {
        is_array = true;
        return *this;
    }
    field& from_context(const context_getter_t& getter_)
    {
        get_from_context = getter_;
        return *this;
    }
    field& use_if(const context_matcher_t& matcher_)
    {
        context_matcher = matcher_;
        return *this;
    }
    field& default_to(const json_value& value)
    {
        default_value = value;
        return *this;
    }
    field& transform(const transformation_t& transformation_)
    {
        transformation = transformation_;
        return *this;
    }
    field& stringify()
    {
        is_array ? force_array_to_string = true : force_string = true;
        return *this;
    }
    field& stringify_elements()
    {
        if (is_array) force_string = true;
        return *this;
    }
    field& composite(std::vector<field> fields)
    {
        children = std::move(fields);
        return *this;
    }
};

template <typename T>
inline field const_field(const string& name, const T& value)
{
    return field(name).from_context([value](const auto&) { return value; });
}

namespace {
inline json_value transform(const json_value& value, const field& field)
{
    json_value res = field.transformation ? field.transformation(value) : value;
    if (field.force_string && !res.is_string())
    {
        res = res.stringify();
    }
    return res;
}

template <typename Source>
std::optional<json_value> retrieve_composite_value(const Source& source, const field& field);

inline std::optional<json_value> retrieve_field_value(const json_value& body, const field& field)
{
    if (field.is_composite())
    {
        return retrieve_composite_value(body, field);
    }
    auto& source = field.source.empty() ? field.target : field.source;
    if (source == "*")
    {
        return transform(body, field);
    }
    if (body.has_member(source))
    {
        return transform(body[source], field);
    }
    else if (!field.is_optional)
    {
        return transform(field.default_value, field);
    }
    return std::optional<json_value>();
}

inline std::optional<json_value> retrieve_array_value(
    const std::vector<json_value>& parts,
    const field& field)
{
    json_value field_array(json_type::tarray);
    bool all_values_empty = true;
    for (auto& part : parts)
    {
        auto opt_value = retrieve_field_value(part, field);
        if (opt_value)
        {
            field_array.push_back(*opt_value);
            all_values_empty = false;
        }
        else
        {
            field_array.push_back(field.default_value);
        }
    }
    bool field_value_empty = field_array.empty() || all_values_empty;
    if (field_value_empty && field.is_optional)
    {
        return std::optional<json_value>();
    }
    return field.force_array_to_string ? json_write(field_array) : field_array;
}

inline std::optional<json_value> retrieve_field_value(const field::context& ctx, const field& field)
{
    if (!field.matches_context(ctx))
    {
        return std::optional<json_value>();
    }
    else if (field.is_array)
    {
        return retrieve_array_value(ctx.event.items, field);
    }
    else if (field.is_composite())
    {
        return retrieve_composite_value(ctx, field);
    }
    else if (field.get_from_context)
    {
        auto value = field.get_from_context(ctx);
        if (!is_null(value)) return value;
    }
    else if (field.from_args)
    {
        return retrieve_field_value(ctx.event.args, field);
    }
    else if (!ctx.event.items.empty())
    {
        return retrieve_field_value(ctx.event.items.front(), field);
    }
    return std::optional<json_value>();
}

template <typename Source>
std::optional<json_value> retrieve_composite_value(const Source& source, const field& field)
{
    json_value res;
    for (auto& child : field.children)
    {
        auto child_value = retrieve_field_value(source, child);
        if (child_value)
        {
            res[child.target] = *child_value;
        }
    }
    return res;
}

}
}
