#pragma once

#include <unicode/unistr.h>
#include <yxiva/core/json.h>
#include <boost/algorithm/string/trim.hpp>

namespace yxiva::mailpusher {

namespace detail {
static const char* ELLIPSIS = "\xe2\x80\xa6";
static const int32_t ELLIPSIS_LEN = 3;

inline int32_t copy_with_max_witdh(string& dest, int32_t max_length, icu::UnicodeString& ustr)
{
    // max utf-8 character length is 4 octets
    int32_t count = max_length / 4;
    if (count)
    {
        auto temp = ustr.tempSubString(0, count);
        temp.toUTF8String(dest);
    }
    return count;
}

inline void copy(string& dest, int32_t start, int32_t max_length, icu::UnicodeString& ustr)
{
    const int TEMP_LEN = 4;
    char temp[TEMP_LEN];
    int32_t total_len = static_cast<int32_t>(dest.length());
    for (auto i = start; i < ustr.length() && total_len < max_length; i++)
    {
        auto size = ustr.extract(i, 1, temp, TEMP_LEN);
        if (total_len + size > max_length)
        {
            break;
        }
        total_len += size;
        dest.append(temp, size);
    }
}

inline string trim_and_ellipsis(const string& value, int32_t length)
{
    if (value.length() > static_cast<std::size_t>(length))
    {
        const int32_t max_value_len = length - ELLIPSIS_LEN;
        string trimmed;
        trimmed.reserve(length);
        icu::UnicodeString ustr = icu::UnicodeString::fromUTF8(icu::StringPiece(value));
        int32_t offset = copy_with_max_witdh(trimmed, max_value_len, ustr);
        copy(trimmed, offset, max_value_len, ustr);
        trimmed.append(ELLIPSIS);
        return trimmed;
    }
    else
    {
        return value;
    }
}

inline string extract_nice_from(const json_value& from_arr_json)
{
    if (from_arr_json.type() == json_type::tarray)
    {
        if (from_arr_json.size() > 0)
        {
            auto from_json = from_arr_json[0UL];
            string display_name = json_get(from_json, "displayName", string{});
            string domain = json_get(from_json, "domain", string{});
            string local = json_get(from_json, "local", string{});
            boost::trim(display_name);
            if (display_name.size())
            {
                return display_name;
            }
            else
            {
                return local + "@" + domain;
            }
        }
    }
    return "";
}
}

inline json_value compose_loc_args(
    const json_value& from_json,
    const string& subject,
    const string& firstline,
    const int32_t max_total_length = 1500)
{
    using detail::ELLIPSIS_LEN;
    assert(max_total_length >= 3 * ELLIPSIS_LEN);

    int32_t max_length = max_total_length;
    json_value locargs(json_type::tarray);
    const string from = detail::extract_nice_from(from_json);
    auto trimmed_from = detail::trim_and_ellipsis(from, max_length - 2 * ELLIPSIS_LEN);
    max_length -= static_cast<int32_t>(trimmed_from.length());
    auto trimmed_subject = detail::trim_and_ellipsis(subject, max_length - ELLIPSIS_LEN);
    max_length -= static_cast<int32_t>(trimmed_subject.length());
    auto trimmed_firstline = detail::trim_and_ellipsis(firstline, max_length);
    locargs.push_back(trimmed_from);
    locargs.push_back(trimmed_subject);
    locargs.push_back(trimmed_firstline);
    return locargs;
}

}
