#pragma once

#include "getters.h"
#include "mappers.h"
#include "transformations.h"
#include "destinations.h"
#include "output_filters.h"
#include "field.h"

namespace yxiva::mailpusher {
namespace {
static const auto COMMON_FIELDS = mapper{ field("uid").from_context(get_uid),
                                          field("uname").from_context(get_suid),
                                          field("lcn").from_context(get_lcn) };

static const auto COMMON_FIELDS_APNS = mapper{ field("z").from_context(get_uid),
                                               field("u").from_context(get_suid),
                                               field("lcn").from_context(get_lcn) };

static const auto APNS_SILENT_APS = mapper{ field("aps").composite({
    const_field("content-available", 1),
    const_field("sound", ""),
}) };
}

static const auto INSERT_MAPPERS = notification_mappers{
    // Destination.
    DESTINATION_EXCEPT_MOBILE,
    // Body.
    COMMON_FIELDS +
        mapper{
            field("operation").from_context(get_operation),
            field("sessionKey").from_arg("session_key"),
            field("fid").from_item("fid"),
            field("mids_str").from_item("mid").array().stringify_elements().stringify(),
            field("threadId").from_item("threadId"),
            field("newCount").from_arg("useful_new_messages"),
            field("status").from_item("seen").optional().transform(seen_transformation),
            field("freshCount").from_arg("fresh_count"),
            field("mid").from_item("mid"), // XXX hotfix hub ios notification formatting
            field("loc-args").from_item("*").transform(loc_args_transformation),
            field("counters").from_arg("counters").optional(),
            field("countersNew").from_arg("countersNew").optional(),
            field("avatarUrl").from_item("avatarUrl").optional(),
            field("tab").from_item("tab").optional(),
            field("srcTab").from_item("src_tab").optional(),
            field("envelopes")
                .array()
                .composite({ field("fid"),
                             field("fidType").from_item("folder").transform(extract_folder_code),
                             field("firstline"),
                             field("from"),
                             field("subject"),
                             field("to"),
                             field("types"),
                             field("labels").default_to(json_value(json_type::tarray)),
                             field("labelsInfo").transform(trunc_labels_info),
                             field("mid"),
                             field("date"),
                             field("size"),
                             field("threadId"),
                             field("avatarUrl").optional(),
                             field("srcFid").from_item("src_fid").optional(),
                             field("messageId").from_item("hdr_message_id"),
                             field("tab").optional(),
                             field("srcTab").from_item("src_tab").optional() }) },
    // Keys.
    COMMON_FIELDS +
        mapper{ field("operation").from_context(get_operation),
                field("session_key").from_arg("session_key"),
                const_field("method_id", ""),
                field("fresh_count").from_arg("fresh_count"),
                field("new_messages").from_arg("useful_new_messages"),
                field("fid").from_item("fid"),
                field("fid_type").from_item("folder").transform(extract_folder_code),
                field("firstline").from_item("firstline"),
                field("hdr_from").from_item("from").transform(hdr_transformation),
                field("hdr_to").from_item("to").transform(hdr_transformation),
                field("hdr_message_id").from_item("hdr_message_id"),
                field("hdr_status").from_item("seen").optional().transform(seen_transformation),
                field("hdr_subject").from_item("subject"),
                field("lid").from_item("labels").transform(lid_transformation),
                field("mid").from_item("mid"),
                field("received_date").from_item("date").transform(format_received_date),
                field("sz").from_item("size"),
                field("thread_id").from_item("threadId"),
                field("tab").from_item("tab").optional(),
                field("srcTab").from_item("src_tab").optional() }
};

static const auto INSERT_MAPPERS_ANDROID = notification_mappers{
    DESTINATION_ANDROID,
    // Body.
    COMMON_FIELDS +
        mapper{
            field("operation").from_context(get_operation),
            field("fid").from_item("fid"),
            field("mids").from_item("mid").array().stringify_elements().stringify(),
            field("tid").from_item("threadId"),
            field("new_messages").from_arg("useful_new_messages"),
            field("status").from_item("seen").optional().transform(seen_transformation),
            field("counters").from_arg("countersNew").optional(),
            field("tab").from_item("tab").optional(),
        },
    INSERT_MAPPERS.keys,
    // Platform-specific.
    { const_field("priority", "high") }
};

static const auto INSERT_MAPPERS_APNS_BRIGHT = notification_mappers{
    DESTINATION_APNS_SUBSCRIPTION,
    // Body.
    COMMON_FIELDS_APNS +
        mapper{ field("operation").from_context(get_operation),
                field("fid").from_item("fid"),
                field("tid").from_item("threadId"),
                field("new_messages").from_arg("useful_new_messages"),
                field("m").from_item("mid"),
                field("counters").from_arg("countersNew").optional(),
                field("avatar").from_item("avatarUrl").optional(),
                field("tab").from_item("tab").optional() },
    INSERT_MAPPERS.keys,
    // Platform-specific.
    { field("collapse-id").from_context(get_collapse_id),
      field("aps").composite({
          const_field("category", "M"),
          const_field("mutable-content", 1),
          field("sound").from_context(get_sound),
          field("badge").from_context(get_badge).optional(),
          field("alert")
              .use_if([](auto& ctx) { return get_ios_version(ctx) >= 10; })
              .composite({
                  field("title").from_item("*").transform(alert_title_transformation),
                  field("subtitle").from_item("*").transform(alert_subtitle_transformation),
                  field("body").from_item("*").transform(alert_body_transformation),
              }),
          field("alert")
              .use_if([](auto& ctx) { return get_ios_version(ctx) < 10; })
              .composite({ const_field("loc-key", "p"),
                           field("loc-args").from_item("*").transform(loc_args_transformation) }),
      }) },
    // Tag.
    "bright",
    // Output, if...
    { filter_action_is_bright, not_(marked_read), allowed_fid_type_for_insert }
};

static const auto INSERT_MAPPERS_APNS_SILENT =
    notification_mappers{ DESTINATION_APNS,
                          INSERT_MAPPERS_APNS_BRIGHT.body,
                          INSERT_MAPPERS_APNS_BRIGHT.keys,
                          APNS_SILENT_APS,
                          // Tag.
                          "silent" };

static const auto MOVE_COPY_MAILS_MAPPERS = notification_mappers{
    // Destination.
    DESTINATION_EXCEPT_MOBILE,
    // Body.
    COMMON_FIELDS +
        mapper{
            field("operation").from_context(get_operation),
            field("sessionKey").from_arg("session_key"),
            field("envelopes")
                .array()
                .composite({ field("fid"),
                             field("fidType").from_item("folder").transform(extract_folder_code),
                             field("firstline"),
                             field("from"),
                             field("subject"),
                             field("to"),
                             field("types"),
                             field("labels").default_to(json_value(json_type::tarray)),
                             field("labelsInfo").transform(trunc_labels_info),
                             field("mid"),
                             field("date"),
                             field("size"),
                             field("threadId"),
                             field("avatarUrl").optional(),
                             field("srcFid").from_item("src_fid").optional(),
                             field("messageId").from_item("hdr_message_id"),
                             field("tab").optional(),
                             field("srcTab").from_item("src_tab").optional() }),
            field("mids_str").from_item("mid").array().stringify_elements().stringify(),
            field("fid").from_item("fid") },
    // Keys.
    COMMON_FIELDS +
        mapper{ field("operation").from_context(get_operation),
                field("session_key").from_arg("session_key"),
                const_field("method_id", ""),
                field("fid").from_item("fid"),
                field("mids").from_item("mid").array().stringify_elements() }
};

static const auto MOVE_COPY_MAILS_MAPPERS_ANDROID = notification_mappers{
    DESTINATION_ANDROID,
    // Body.
    COMMON_FIELDS +
        mapper{ field("operation").from_context(get_operation),
                field("mids").from_item("mid").array().stringify_elements().stringify(),
                field("fid").from_item("fid") },
    MOVE_COPY_MAILS_MAPPERS.keys
};

static const auto MOVE_COPY_MAILS_MAPPERS_APNS =
    notification_mappers{ DESTINATION_APNS,
                          // Body.
                          COMMON_FIELDS_APNS +
                              mapper{ field("operation").from_context(get_operation),
                                      field("fid").from_item("fid"),
                                      field("tab").from_item("tab").optional() },
                          MOVE_COPY_MAILS_MAPPERS.keys,
                          APNS_SILENT_APS };

static const auto FAKE_MOVE_DELETED_MAILS_MAPPERS = notification_mappers{
    // Destination.
    DESTINATION_EXCEPT_MOBILE,
    // Body.
    COMMON_FIELDS +
        mapper{ const_field("operation", xiva_operation_name(action::MOVE_MAILS)),
                const_field("fid", -1),
                field("sessionKey").from_arg("session_key"),
                field("mids_str").from_item("mid").array().stringify_elements().stringify() },
    // Keys.
    COMMON_FIELDS +
        mapper{ const_field("operation", xiva_operation_name(action::MOVE_MAILS)),
                const_field("fid", -1),
                field("session_key").from_arg("session_key"),
                const_field("method_id", ""),
                field("mids").from_item("mid").array().stringify_elements() }
};

static const auto FAKE_MOVE_DELETED_MAILS_MAPPERS_ANDROID = notification_mappers{
    DESTINATION_ANDROID,
    // Body.
    COMMON_FIELDS +
        mapper{ const_field("operation", xiva_operation_name(action::MOVE_MAILS)),
                const_field("fid", -1),
                field("mids").from_item("mid").array().stringify_elements() },
    FAKE_MOVE_DELETED_MAILS_MAPPERS.keys
};

static const auto FAKE_MOVE_DELETED_MAILS_MAPPERS_APNS =
    notification_mappers{ DESTINATION_APNS,
                          // Body.
                          COMMON_FIELDS_APNS +
                              mapper{
                                  const_field("operation", xiva_operation_name(action::MOVE_MAILS)),
                                  const_field("fid", -1),
                              },
                          FAKE_MOVE_DELETED_MAILS_MAPPERS.keys,
                          APNS_SILENT_APS };

static const auto DELETE_MAILS_MAPPERS =
    notification_mappers{ // Destination.
                          DESTINATION_EXCEPT_MOBILE,
                          // Body.
                          COMMON_FIELDS +
                              mapper{ field("operation").from_context(get_operation),
                                      field("sessionKey").from_arg("session_key"),
                                      field("mids").from_item("mid").array().stringify_elements() },
                          // Keys.
                          { field("operation").from_context(get_operation),
                            field("session_key").from_arg("session_key"),
                            field("lcn").from_context(get_lcn) }
    };

static const auto MARK_UNMARK_MAILS_MAPPERS = notification_mappers{
    // Destination.
    DESTINATION_EXCEPT_MOBILE,
    // Body.
    COMMON_FIELDS +
        mapper{ field("operation").from_context(get_operation),
                field("sessionKey").from_arg("session_key"),
                field("mids_str").from_item("mid").array().stringify_elements().stringify(),
                field("m_lids_str")
                    .from_arg("labels")
                    .default_to(json_value(json_type::tarray))
                    .optional()
                    .stringify(),
                field("all_labels_str")
                    .from_item("labels")
                    .default_to(json_value(json_type::tarray))
                    .array()
                    .stringify()
                    .optional(),
                field("fids_str")
                    .from_item("fid")
                    .array()
                    .stringify_elements()
                    .stringify()
                    .optional() },
    // Keys.
    COMMON_FIELDS +
        mapper{
            field("operation").from_context(get_operation),
            field("session_key").from_arg("session_key"),
            const_field("method_id", ""),
            field("m_lids").from_arg("labels").default_to(json_value(json_type::tarray)).optional(),
            field("all_labels")
                .from_item("labels")
                .default_to(json_value(json_type::tarray))
                .array()
                .optional(),
            field("fids").from_item("fid").array().stringify_elements().optional(),
            field("tids").from_item("tid").array().stringify_elements().optional(),
            field("mids").from_item("mid").array().stringify_elements() }
};

static const auto MARK_UNMARK_MAILS_MAPPERS_ANDROID = notification_mappers{
    DESTINATION_ANDROID,
    // Body.
    COMMON_FIELDS +
        mapper{
            field("operation").from_context(get_operation),
            field("mids").from_item("mid").array().stringify_elements(),
            field("m_lids")
                .from_arg("labels")
                .default_to(json_value(json_type::tarray))
                .optional()
                .stringify(),
            field("all_labels")
                .from_item("labels")
                .default_to(json_value(json_type::tarray))
                .array()
                .stringify()
                .optional(),
            field("fids").from_item("fid").array().stringify().stringify_elements().optional() },
    MARK_UNMARK_MAILS_MAPPERS.keys
};

static const auto MARK_UNMARK_MAILS_MAPPERS_APNS =
    notification_mappers{ DESTINATION_APNS,
                          // Body.
                          COMMON_FIELDS_APNS +
                              mapper{
                                  field("operation").from_context(get_operation),
                                  field("m_lids")
                                      .from_arg("labels")
                                      .default_to(json_value(json_type::tarray))
                                      .optional()
                                      .stringify(),
                              },
                          MARK_UNMARK_MAILS_MAPPERS.keys,
                          APNS_SILENT_APS };

static const auto UPDATE_LABELS_MAPPERS = notification_mappers{
    // Destination.
    DESTINATION_EXCEPT_MOBILE,
    // Body.
    COMMON_FIELDS +
        mapper{ const_field("operation", "update labels"),
                field("sessionKey").from_arg("session_key"),
                field("labelsAdd").optional().from_context([](auto& ctx) {
                    return extract_labels_from_args(ctx.event.args, "lids_add", true);
                }),
                field("labelsDel").optional().from_context([](auto& ctx) {
                    return extract_labels_from_args(ctx.event.args, "lids_del", false);
                }),
                field("status").from_arg("flags").transform(flags_transformation),
                field("status").from_arg("seen").optional().transform(seen_transformation),
                field("fids").from_item("fid").array().stringify_elements().optional(),
                field("tids").from_item("tid").array().stringify_elements().optional(),
                field("newCount").from_arg("useful_new_messages"),
                field("mids").from_item("mid").array().stringify_elements() },
    // Keys.
    { const_field("operation", "update labels"),
      field("session_key").from_arg("session_key"),
      field("lcn").from_context(get_lcn) }
};

static const auto UPDATE_LABELS_MAPPERS_APNS = notification_mappers{
    DESTINATION_APNS,
    // Body.
    COMMON_FIELDS_APNS +
        mapper{ const_field("operation", "update labels"),
                field("new_messages").from_arg("useful_new_messages"),
                field("fids").from_item("fid").array().stringify_elements().optional(),
                field("tabs").from_item("tab").array().stringify_elements().optional(),
                field("mids").from_item("mid").array().stringify_elements() },
    UPDATE_LABELS_MAPPERS.keys,
    APNS_SILENT_APS
};

static const auto STATUS_CHANGE_MAPPERS = notification_mappers{
    // Destination.
    DESTINATION_EXCEPT_MOBILE,
    // Body.
    COMMON_FIELDS +
        mapper{ field("operation").from_context(get_operation),
                field("sessionKey").from_arg("session_key"),
                field("mids_str").from_item("mid").array().stringify_elements().stringify(),
                field("new_messages").from_arg("useful_new_messages"),
                field("status").from_arg("flags").transform(flags_transformation).optional(),
                field("status").from_arg("seen").optional().transform(seen_transformation) },
    // Keys.
    COMMON_FIELDS +
        mapper{ field("operation").from_context(get_operation),
                field("session_key").from_arg("session_key"),
                const_field("method_id", ""),
                field("new_messages").from_arg("useful_new_messages"),
                field("status").from_arg("flags").transform(flags_transformation).optional(),
                field("status").from_arg("seen").optional().transform(seen_transformation),
                field("mids").from_item("mid").array().stringify_elements() }
};

static const auto STATUS_CHANGE_MAPPERS_ANDROID = notification_mappers{
    DESTINATION_ANDROID,
    // Body.
    COMMON_FIELDS +
        mapper{ field("operation").from_context(get_operation),
                field("mids").from_item("mid").array().stringify_elements().stringify(),
                field("status").from_arg("flags").transform(flags_transformation).optional(),
                field("status").from_arg("seen").optional().transform(seen_transformation) },
    STATUS_CHANGE_MAPPERS.keys
};

static const auto STATUS_CHANGE_MAPPERS_APNS = notification_mappers{
    DESTINATION_APNS,
    // Body.
    COMMON_FIELDS_APNS +
        mapper{ field("operation").from_context(get_operation),
                field("mids").from_item("mid").array().stringify_elements().stringify(),
                field("status").from_arg("flags").transform(flags_transformation).optional(),
                field("status").from_arg("seen").optional().transform(seen_transformation) },
    STATUS_CHANGE_MAPPERS.keys,
    APNS_SILENT_APS
};

static const auto UPDATE_LABELS_STATUS_CHANGE_MAPPERS = notification_mappers{
    DESTINATION_EXCEPT_MOBILE,
    UPDATE_LABELS_MAPPERS.body +
        field("mids_str").from_item("mid").array().stringify_elements().stringify(),
    UPDATE_LABELS_MAPPERS.keys
};

static const auto RESET_FRESH_MAPPERS =
    notification_mappers{ // Destination.
                          DESTINATION_EXCEPT_MOBILE,
                          // Body.
                          COMMON_FIELDS +
                              mapper{
                                  field("operation").from_context(get_operation),
                                  field("sessionKey").from_arg("session_key"),
                              },
                          // Keys.
                          COMMON_FIELDS +
                              mapper{ field("operation").from_context(get_operation),
                                      field("session_key").from_arg("session_key"),
                                      const_field("method_id", "") }
    };

static const auto TRANSFER_MAPPERS =
    notification_mappers{ // Destination.
                          DESTINATION_EXCEPT_MOBILE,
                          // Body.
                          COMMON_FIELDS +
                              mapper{ field("operation").from_context(get_operation),
                                      field("sessionKey").from_arg("session_key"),
                                      field("event_ts") },
                          // Keys.
                          COMMON_FIELDS +
                              mapper{ field("operation").from_context(get_operation),
                                      field("session_key").from_arg("session_key"),
                                      const_field("method_id", "") }
    };

static const auto TRANSFER_MAPPERS_ANDROID =
    notification_mappers{ DESTINATION_ANDROID,
                          // Body.
                          COMMON_FIELDS +
                              mapper{
                                  field("operation").from_context(get_operation),
                              },
                          TRANSFER_MAPPERS.keys };

static const auto TRANSFER_MAPPERS_APNS =
    notification_mappers{ DESTINATION_APNS,
                          // Body.
                          COMMON_FIELDS_APNS +
                              mapper{
                                  field("operation").from_context(get_operation),
                              },
                          TRANSFER_MAPPERS.keys,
                          APNS_SILENT_APS };
}