#include "json_helpers.h"

namespace yxiva::mailpusher {

operation::result decode_event(event& event, const json_value& json_event)
{
    // TODO Use required fields checker from webui code.
    auto action_str = json_get(json_event, "action", string{});
    event.action_type = resolve_action(action_str);
    if (event.action_type == action::UNKNOWN)
    {
        return "action is not supported: " + action_str;
    }

    if (!json_event.has_member("change_id") || !json_event["change_id"].is_uint64())
    {
        return "missing or invalid member " + string("change_id");
    }
    event.change_id = json_get(json_event, "change_id", 0);

    if (!json_event.has_member("items") || !json_event["items"].is_array())
    {
        return "missing or invalid member " + string("items");
    }
    event.items.reserve(json_event["items"].size());
    for (auto&& item : json_event["items"].array_items())
    {
        if (!item.is_object()) return "unexpected element type in " + string("items");
        event.items.push_back(item);
    }

    if (!json_event.has_member("args") || !json_event["args"].is_object())
    {
        return "missing or invalid member " + string("args");
    }
    event.args = json_event["args"];

    if (!json_event.has_member("ts") || !json_event["ts"].is_uint64())
    {
        return "missing or invalid member " + string("ts");
    }
    event.ts = json_get(json_event, "ts", 0UL);

    event.lcn = json_get(json_event, "lcn", string{});
    event.x_request_id = json_get(json_event, "request_id", string{});

    // Account for cached metadata present in request.
    if (json_event.has_member("status"))
    {
        auto&& status = json_event["status"];
        event.metadata_count = json_get(status, "metadata_count", 0U);
        event.avatar_fetched =
            json_get(status, "avatar_fetched", false) ? fetch_status::FETCHED : fetch_status::NONE;
        if (status["sent_notification_ids"].is_array())
        {
            auto&& sent_notification_ids = status["sent_notification_ids"];
            for (auto&& step : sent_notification_ids.array_items())
            {
                event.sent_notification_ids.push_back(step.to_string());
            }
        }
    }
    return operation::success;
}

operation::result parse_task(task& task, const string& raw_events)
{
    json_value json_task;
    if (auto error = json_task.parse(raw_events))
    {
        return "invalid json: " + *error;
    }

    task.uid = json_get(json_task, "uid", string{});
    task.suid = json_get(json_task, "suid", string{});
    task.db = json_get(json_task, "dbname", string{});
    // TODO Need both?
    if (task.uid.empty() && task.suid.empty())
    {
        return "missing user info";
    }

    auto&& json_events = json_task["events"];
    task.events.reserve(json_events.size());
    for (auto&& json_event : json_events.array_items())
    {
        task.events.emplace_back();
        auto& event = task.events.back();
        event.uid = task.uid;
        event.suid = task.suid;
        if (auto res = decode_event(event, json_event); !res)
        {
            return res;
        }
    }
    return operation::success;
}

json_value dump_task(const task& task)
{
    json_value result;
    result["processed"] = task.processed;
    auto&& events = result["events"];
    events.set_array();
    for (auto i = task.processed; i < task.events.size(); ++i)
    {
        auto&& event = task.events[i];
        json_value json_event(json_type::tobject);
        auto&& status = json_event["status"];
        status["metadata_count"] = event.metadata_count;
        status["avatar_fetched"] = (event.avatar_fetched == fetch_status::FETCHED);
        for (auto& sent_id : event.sent_notification_ids)
        {
            status["sent_notification_ids"].push_back(sent_id);
        }
        auto&& items = json_event["items"];
        items.set_array();
        // Only put records which have metadata.
        for (auto k = 0UL; k < event.metadata_count; ++k)
        {
            items.push_back(event.items[k]);
        }
        events.push_back(json_event);
    }
    return result;
}

}
