#pragma once

#include "settings.h"
#include "../mod_log/mod_log.h"
#include <mailpusher/task.h>
#include <mailpusher/types.h>
#include <mailpusher/errors.h>
#include <ymod_ratecontroller/rate_controller.h>
#include <yplatform/coroutine.h>
#include <yplatform/util/safe_call.h>
#include <boost/range/adaptors.hpp>
#include <algorithm>
#include <map>
#include <set>
#include <vector>

#include "processor_def.h"

namespace yxiva::mailpusher {

template <typename HttpClient>
struct metadata_fetcher
{
    using yield_context_t = yplatform::yield_context<metadata_fetcher<HttpClient>>;
    struct job
    {
        job(event& event, json_value& item, const string& mid) : event(event), item(item), mid(mid)
        {
        }

        event& event;
        json_value& item;
        string mid;
    };

    shared_ptr<task> task;
    ymod_ratecontroller::rate_controller_ptr rate_controller;
    HttpClient& http_client;
    const settings& settings;
    callback_t cb;

    ymod_ratecontroller::completion_handler rate_controller_on_complete{};
    error_code error{};
    std::vector<job> jobs = {};
    json_value json_metadata{};
    std::map<string, json_value> envelopes_by_mid{};
    // Current processing step.
    enum
    {
        FETCH_REPLICA,
        FETCH_MASTER,
        DONE,
        FAILED
    } step = FETCH_REPLICA;

    static const string& name()
    {
        static const string NAME = "meta";
        return NAME;
    }

    void operator()(
        yield_context_t yield_context,
        const error_code& ec = {},
        yhttp::response resp = {})
    {
        try
        {
            reenter(yield_context)
            {
                yield rate_controller->post(yield_context, "", task->deadline());
                if (error = error::from_rate_controller_error(ec))
                {
                    yield break;
                }
                // If no events with mids were generated, consider things done.
                if (!prepare_jobs()) step = DONE;
                // Up to 2 http calls – replica and master requests.
                while (step != DONE && step != FAILED)
                {
                    yield do_request(yield_context);
                    // Break without yield in case of error to terminate loop
                    // and immediately reach coroutine end.
                    if (error = error::from_http_response(ec, resp))
                    {
                        if (ec)
                        {
                            LOG_ERROR("http_error", ec.message());
                        }
                        else
                        {
                            LOG_ERROR("http_response", "code " + std::to_string(resp.status));
                        }
                        step = FAILED;
                    }
                    else if (error = process_response(resp))
                    {
                        step = FAILED;
                    }
                }
                if (step == DONE)
                {
                    fill_events();
                }
            }
        }
        catch (const std::exception& ex)
        {
            LOG_ERROR("exception", ex.what());
            error = make_error(error::internal_error);
        }

        if (yield_context.is_complete())
        {
            cb(error);
            yplatform::safe_call(rate_controller_on_complete);
        }
    }

    string make_mids_param()
    {
        using boost::adaptors::transformed;
        using boost::adaptors::filtered;
        static const auto mid_getter = [](auto& job) { return job.mid; };
        return boost::algorithm::join(
            jobs | filtered([](auto& job) { return job.mid.size(); }) | transformed(mid_getter),
            "&mids=");
    }

    void do_request(yield_context_t yield_context)
    {
        // Check if we made request before.
        http_client.async_run(
            task,
            yhttp::request::GET(
                settings.meta.request +
                yhttp::url_encode({ { "uid", task->uid },
                                    { "dbtype", step == FETCH_MASTER ? "master" : "replica" },
                                    { "full_folders_and_labels", "1" },
                                    { "mids", "" } }) +
                make_mids_param()),
            yield_context);
    }

    // Rate controller handler.
    void operator()(
        yield_context_t yield_context,
        const error_code& ec,
        ymod_ratecontroller::completion_handler on_complete)
    {
        rate_controller_on_complete = std::move(on_complete);
        (*this)(yield_context, ec);
    }

    error_code process_response(const yhttp::response& resp)
    {
        json_metadata = json_value();
        // TODO Skip all events in case of some unretriable fail?
        if (auto res = json_parse(json_metadata, resp.body); !res)
        {
            LOG_ERROR("malformed_response", res.error_reason);
            return make_error(error::bad_gateway);
        }
        if (json_metadata.has_member("error"))
        {
            LOG_ERROR("bad_response", json_write(json_metadata["error"]));
            return make_error(
                json_get(json_metadata["error"], "code", -1) == 1006 ? error::uninitialized_user :
                                                                       error::bad_gateway);
        }
        if (!json_metadata.has_member("envelopes") || !json_metadata["envelopes"].is_array())
        {
            LOG_ERROR("malformed_response", "missing or invalid envelopes");
            return make_error(error::bad_gateway);
        }
        envelopes_by_mid = map_envelopes(json_metadata["envelopes"]);
        // Determine next step.
        if (step == FETCH_MASTER)
        {
            step = DONE;
        }
        else if (step == FETCH_REPLICA)
        {
            step = need_more_metadata() ? FETCH_MASTER : DONE;
        }
        else
        {
            // If reaches this branch, something is not right.
            LOG_ERROR("unexpected_state", "step " + std::to_string(step));
            step = FAILED;
        }
        return make_error(error::success);
    }

    size_t prepare_jobs()
    {
        size_t mids_count = 0;
        // No reserve because it's difficult to estimate appropriate size.
        for (auto i = task->processed; i < task->events.size() &&
             mids_count < settings.meta.max_mids_in_filter_search_request;
             ++i)
        {
            auto& event = task->events[i];
            if (!required(event)) continue;
            for (auto j = event.metadata_count; j < event.items.size() &&
                 mids_count < settings.meta.max_mids_in_filter_search_request;
                 ++j)
            {
                auto& item = event.items[j];
                auto mid = json_get<string>(item, "mid", "");
                jobs.emplace_back(event, item, mid);
                if (mid.size()) ++mids_count;
            }
        }
        return mids_count;
    }

    static std::map<string, json_value> map_envelopes(const json_value_ref& envelopes)
    {
        std::map<string, json_value> envelopes_map;
        for (auto&& envelope : envelopes.array_items())
        {
            if (auto mid = json_get<string>(envelope, "mid", ""); mid.size())
            {
                envelopes_map.emplace(mid, envelope);
            }
        }
        return envelopes_map;
    }

    void fill_events()
    {
        for (auto& job : jobs)
        {
            if (job.mid.empty())
            {
                ++job.event.metadata_count;
            }
            else if (auto it = envelopes_by_mid.find(job.mid); it == envelopes_by_mid.end())
            {
                // Read from master have failed or is impossible
                ++job.event.metadata_count;
                // For insert, don't send push without metadata.
                if (job.event.action_type == action::NEW_MAIL) job.event.skip = true;
            }
            else
            {
                json_merge(job.item, it->second);
                ++job.event.metadata_count;
            }
        }
    }

    bool need_more_metadata()
    {
        // There are jobs which envelopes are not fetched and lag is small enough.
        auto expired = [this](auto& job) {
            return settings.meta.fallback_to_db_master_max_lag &&
                job.event.lag() > settings.meta.fallback_to_db_master_max_lag;
        };
        auto missing_envelope = [this](auto& job) {
            return job.mid.size() && !envelopes_by_mid.count(job.mid);
        };
        return std::none_of(jobs.begin(), jobs.end(), expired) &&
            std::any_of(jobs.begin(), jobs.end(), missing_envelope);
    }

    static bool required(const event& ev)
    {
        static const std::set<action> actions_with_metadata{
            action::NEW_MAIL, action::MOVE_MAILS, action::MOVE_MAILS2, action::COPY_MAILS
        };
        return !ev.skip && ev.metadata_count < ev.items.size() &&
            actions_with_metadata.count(ev.action_type);
    }

    static bool done(const event& ev)
    {
        return ev.metadata_count == ev.items.size() && !ev.skip;
    }
};

}

#undef LOG_ERROR
#include <boost/asio/unyield.hpp>
