#pragma once

#include "settings.h"
#include "processor.h"
#include "batch_push.h"
#include <mailpusher/task.h>
#include <mailpusher/types.h>
#include <ymod_webserver/server.h>
#include <yplatform/module.h>
#include <yplatform/find.h>
#include <map>
#include <stdexcept>

namespace yxiva::mailpusher {

class module : public yplatform::module
{
    using processor = processor<yhttp::cluster_client>;
    using http_clients_map = std::map<string, yhttp::cluster_client>;
    using mod_rc = ymod_ratecontroller::rate_controller_module;

public:
    module(yplatform::reactor& reactor, const yplatform::ptree& conf)
        : reactor_(reactor), settings_(conf), http_clients_(std::make_shared<http_clients_map>())
    {
        create_http_for_all_environments(processor::subscription_fetcher_t::name(), settings_.list);
        create_http_for_all_environments(processor::counters_fetcher_t::name(), settings_.counters);
        create_http_client(processor::metadata_fetcher_t::name(), settings_.meta.http);
        create_http_client(processor::avatar_fetcher_t::name(), settings_.ava.http);
        create_http_client(processor::notification_sender_t::name(), settings_.send.http);
        create_http_client(processor::searchapp_sender_t::name(), settings_.searchapp.http);

        auto web_server = yplatform::find<ymod_webserver::server>("web_server");
        web_server->bind("", { "/ping" }, [](auto stream) {
            stream->result(ymod_webserver::codes::ok, "pong");
        });
        web_server->bind("", { "/batch_push" }, batch_push{});
    }

    void process_task(const shared_ptr<task>& task, callback_t cb)
    {
        yplatform::spawn(processor{ task,
                                    settings_,
                                    http_clients_,
                                    yplatform::find<mod_rc, std::shared_ptr>("rate_controller"),
                                    cb,
                                    yplatform::find<mod_log, std::shared_ptr>("mod_log") });
    }

private:
    void create_http_client(const string& name, const yhttp::cluster_client::settings& s)
    {
        if (auto res = http_clients_->try_emplace(name, reactor_, s); !res.second)
        {
            throw std::logic_error("duplicate http client");
        }
    }

    template <typename SettingsMap>
    void create_http_for_all_environments(const string& name, const SettingsMap& settings_map)
    {
        for (auto& [environment, settings] : settings_map)
        {
            create_http_client(name + "." + environment, settings.http);
        }
    }

    yplatform::reactor& reactor_;
    settings settings_;
    std::shared_ptr<http_clients_map> http_clients_;
};

}
