#pragma once

#include "format/mappers.h"
#include "format/destinations.h"
#include "settings.h"
#include <mailpusher/notification.h>
#include <mailpusher/task.h>
#include <mailpusher/event.h>
#include <yxiva/core/platforms.h>
#include <yplatform/log.h>
#include <boost/algorithm/string/join.hpp>

namespace yxiva::mailpusher {
namespace hacks {

inline bool roll_rtec_3674(const string& uid, unsigned rollout_percent)
{
    return std::strtoull(uid.c_str(), nullptr, 10) % 100 < rollout_percent;
}

}

class notification_factory
{
public:
    std::vector<notification> make(
        const shared_ptr<task>& task,
        const event& event,
        const settings& settings) const
    {
        return hacks::roll_rtec_3674(task->uid, settings.send.rtec_3674_rollout_percent) ?
            make(task, event, MAPPERS) :
            make(task, event, PRE_RTEC_3674_MAPPERS);
    }

    std::vector<notification> make(
        const shared_ptr<task>& task,
        const event& event,
        const mappers_collection& mappers_by_action) const
    {
        try
        {
            auto action_mappers = mappers_by_action.find(event.action_type);
            if (action_mappers == mappers_by_action.end())
            {
                throw std::runtime_error("unsupported action type");
            }

            std::vector<notification> notifications;
            for (auto& subscription : task->subscriptions)
            {
                for (const auto& mappers : action_mappers->second)
                {
                    add_notification(event, subscription, mappers, notifications);
                }
            }
            return notifications;
        }
        catch (const std::exception& ex)
        {
            YLOG_CTX_GLOBAL(task, error)
                << "error while formatting notification: exception=\"" << ex.what() << "\"";
            return { {} };
        }
    }

private:
    void add_notification(
        const event& event,
        const subscription& subscription,
        const notification_mappers& mappers,
        std::vector<notification>& notifications) const
    {
        auto destination = fill_placeholders(mappers.destination, subscription);
        if (!matches(subscription, destination))
        {
            return;
        }
        auto keys = stringify_values(apply_mapper(mappers.keys, event, subscription));
        auto operation = json_get<string>(keys, "operation", "");
        auto id = make_notification_id(operation, destination, mappers.tag);
        if (already_formatted(id, notifications))
        {
            return;
        }
        auto notification = make_notification(
            id,
            event,
            subscription,
            destination,
            mappers.body,
            std::move(keys),
            mappers.platform_specific);
        if (mappers.output(notification, subscription))
        {
            notifications.emplace_back(std::move(notification));
        }
    }

    destination fill_placeholders(const destination& destination, const subscription& subscription)
        const
    {
        auto ret = destination;
        auto& subscription_id = destination.subscription_id;
        if (subscription_id.size() == 1 && *subscription_id.begin() == SUBSCRIPTION_ID_PLACEHOLDER)
        {
            ret.subscription_id = { subscription.common()->id };
        }
        return ret;
    }

    notification make_notification(
        const string& id,
        const event& event,
        const subscription& subscription,
        const destination& destination,
        const mapper& body_mapper,
        json_value keys,
        const mapper& platform_specific_mapper) const
    {
        auto body = apply_mapper(body_mapper, event, subscription);
        auto delivery = subscription.mobile() ? notification::delivery_mode::direct :
                                                notification::delivery_mode::queued;
        auto repack = json_value{};
        if (subscription.mobile())
        {
            repack =
                make_repack(destination.platform, platform_specific_mapper, event, subscription);
        }
        bool silent = is_silent(repack);
        return {
            id, destination, delivery, silent, std::move(body), std::move(keys), std::move(repack)
        };
    }

    // Keys must all be strings, because in xiva keys is
    // a map<string, string>.
    json_value stringify_values(const json_value& keys) const
    {
        json_value ret;
        for (auto it = keys.members_begin(); it != keys.members_end(); ++it)
        {
            string key(it.key());
            // Stringify adds quotes around string values,
            // e.g. foo becomes "foo". We want string values
            // without quotes. And we cannot use to_string(),
            // because keys might be arrays or objects, and
            // to_string throws for those.
            if ((*it).is_string())
            {
                ret[key] = *it;
            }
            else
            {
                ret[key] = (*it).stringify();
            }
        }
        return ret;
    }

    string make_notification_id(
        const string& operation,
        const destination& destination,
        const string& tag) const
    {
        static const std::size_t SUBSCRIPTION_ID_LEN = 10;
        std::vector<string> truncated_subscription_ids;
        truncated_subscription_ids.reserve(destination.subscription_id.size());
        std::transform(
            destination.subscription_id.begin(),
            destination.subscription_id.end(),
            std::back_inserter(truncated_subscription_ids),
            [](auto& s) { return s.substr(0, SUBSCRIPTION_ID_LEN); });
        return boost::algorithm::join_if(
            std::vector<string>{ operation,
                                 boost::algorithm::join(destination.transport, " "),
                                 boost::algorithm::join(destination.platform, " "),
                                 boost::algorithm::join(truncated_subscription_ids, " "),
                                 tag },
            " ",
            [](auto& s) { return !s.empty(); });
    }

    inline json_value make_repack(
        const std::set<string>& platforms,
        const mapper& platform_specific,
        const event& event,
        const subscription& subscription) const
    {
        json_value repack;
        for (auto&& platform : platforms)
        {
            repack[platform] = apply_mapper(platform_specific, event, subscription);
            auto&& payload = repack[platform]["repack_payload"];
            payload.push_back("*");
            payload.push_back()["transit-id"] = "::xiva::transit_id";
        }
        return repack;
    }

    bool is_silent(const json_value& repack) const
    {
        return repack[platform::APNS]["aps"]["content-available"] == 1;
    }

    bool matches(const subscription& subscription, const destination& destination) const
    {
        return (destination.transport.empty() ||
                destination.transport.count(subscription.transport()) > 0) &&
            (destination.platform.empty() ||
             subscription.mobile() &&
                 destination.platform.count(subscription.mobile()->platform) > 0);
    }

    bool already_formatted(const string& id, const std::vector<notification>& notifications) const
    {
        return notifications.end() !=
            std::find_if(notifications.begin(), notifications.end(), [&id](const notification& n) {
                   return n.id == id;
               });
    }
};

}
