#pragma once

#include "settings.h"
#include "../mod_log/mod_log.h"
#include "format/extract_folder_code.h"
#include "format/mappers.h"
#include <mailpusher/notification.h>
#include <mailpusher/task.h>
#include <mailpusher/types.h>
#include <mailpusher/errors.h>
#include <ymod_ratecontroller/rate_controller.h>
#include <yplatform/coroutine.h>
#include <yplatform/util/safe_call.h>

#include "processor_def.h"

namespace yxiva::mailpusher {

template <typename HttpClient, typename NotificationFactory>
struct notification_sender
{
    using notification_factory_type = NotificationFactory;
    using yield_context_t = yplatform::yield_context<notification_sender>;

    shared_ptr<task> task;
    ymod_ratecontroller::rate_controller_ptr rate_controller;
    HttpClient& http_client;
    const settings& settings;
    callback_t cb;
    // TODO: move factory to processor, does not look like
    // sender's responsibility.
    notification_factory_type notification_factory{};

    ymod_ratecontroller::completion_handler rate_controller_on_complete{};
    std::vector<notification> notifications{};
    error_code error{};
    size_t index = 0;
    yhttp::request notification_request{};

    static const string& name()
    {
        static const string NAME = "xiva_send";
        return NAME;
    }

    void operator()(
        yield_context_t yield_context,
        const error_code& ec = {},
        yhttp::response resp = {})
    {
        try
        {
            reenter(yield_context)
            {
                yield rate_controller->post(yield_context, "", task->deadline());
                if (error = error::from_rate_controller_error(ec))
                {
                    yield break;
                }

                notifications =
                    notification_factory.make(task, task->next_unprocessed_event(), settings);
                for (; index < notifications.size(); ++index)
                {
                    if (already_sent(notifications[index]))
                    {
                        YLOG_CTX_GLOBAL(task, info) << "notification " << notifications[index].id
                                                    << " skipped (sent on previous try)";
                        continue;
                    }
                    if (error = prepare_request(notifications[index]))
                    {
                        yield break;
                    }
                    yield http_client.async_run(task, notification_request, yield_context);
                    // Log first notification's context and transitid.
                    if (auto it = resp.headers.find("y-context");
                        index == 0 && it != resp.headers.end())
                    {
                        task->next_unprocessed_event().xiva_context = it->second;
                    }
                    if (auto it = resp.headers.find("transitid");
                        index == 0 && it != resp.headers.end())
                    {
                        task->next_unprocessed_event().xiva_transit = it->second;
                    }
                    if (error = error::from_http_response(ec, resp))
                    {
                        if (ec)
                        {
                            LOG_ERROR("http_error", ec.message());
                        }
                        else
                        {
                            LOG_ERROR("http_response", "code " + std::to_string(resp.status));
                        }
                        yield break;
                    }
                    task->next_unprocessed_event().sent_notification_ids.push_back(
                        notifications[index].id);
                }
            }
        }
        catch (const std::exception& ex)
        {
            LOG_ERROR("exception", ex.what());
            error = make_error(error::internal_error);
        }

        if (yield_context.is_complete())
        {
            cb(error);
            yplatform::safe_call(rate_controller_on_complete);
        }
    }

    bool already_sent(const notification& notification)
    {
        auto& steps = task->next_unprocessed_event().sent_notification_ids;
        return std::find(steps.begin(), steps.end(), notification.id) != steps.end();
    }

    error_code prepare_request(const notification& notification)
    {
        notification_request = yhttp::request::POST(
            settings.send.request +
                yhttp::url_encode({
                    { "user", task->uid },
                    { "service", "mail" },
                    { "event", notification.operation() },
                    { "request_id", make_request_id(notification) },
                    { "source", "mailpusher" },
                    { "session", notification.session_key() },
                    { "ttl", (notification.silent ? 0 : settings.send.ttl) },
                }),
            make_headers(notification),
            make_body(notification));
        // Drop notifications larger than 1 Mb.
        if (notification_request.body->size() > 1048576)
        {
            return make_error(error::payload_too_large);
        }
        return make_error(error::success);
    }

    string make_request_id(const notification& notification)
    {
        auto ret = task->uniq_id() + "_" + notification.id;
        std::replace_if(
            ret.begin(), ret.end(), [](char c) { return !isalnum(c); }, '_');
        return ret;
    }

    string make_body(const notification& notification)
    {
        json_value body;
        body["payload"] = notification.body;
        body["keys"] = notification.keys;
        if (auto subscriptions = make_subscription_filters(notification.destination))
        {
            body["subscriptions"] = std::move(subscriptions);
        }
        if (notification.repack)
        {
            body["repack"] = notification.repack;
        }
        return json_write(body);
    }

    string make_headers(const notification& n)
    {
        string headers = "";
        if (n.delivery == notification::delivery_mode::direct)
        {
            headers += "X-DeliveryMode: direct\r\n";
        }
        return headers;
    }

    json_value make_subscription_filters(const destination& destination)
    {
        json_value subscriptions;
        add_set_filter(subscriptions, "transport", destination.transport);
        add_set_filter(subscriptions, "platform", destination.platform);
        add_set_filter(subscriptions, "subscription_id", destination.subscription_id);
        return subscriptions;
    }

    template <typename Set>
    void add_set_filter(json_value& subscriptions, const string& name, Set& values)
    {
        if (values.empty())
        {
            return;
        }
        auto&& filter = subscriptions.push_back()[name];
        for (auto& v : values)
        {
            filter.push_back(v);
        }
    }

    // Rate controller handler.
    void operator()(
        yield_context_t yield_context,
        const error_code& ec,
        ymod_ratecontroller::completion_handler on_complete)
    {
        rate_controller_on_complete = std::move(on_complete);
        (*this)(yield_context, ec);
    }

    static bool is_appropriate_insert_folder(
        const struct event& event,
        const struct settings& settings)
    {
        if (event.action_type == action::NEW_MAIL && event.items.size())
        {
            auto folder_code = extract_folder_code(event.items.front()["folder"]);
            return !settings.send.ignored_folder_codes.count(folder_code);
        }
        return true;
    }

    static operation::result required(const struct event& ev, const struct settings& settings)
    {
        if (ev.skip) return "no metadata";
        if (!is_supported(ev.action_type)) return "ignored type";
        if (!is_appropriate_insert_folder(ev, settings)) return "ignored folder";
        return operation::success;
    }
};

}

#include "processor_undef.h"