#pragma once

#include <mailpusher/account.h>
#include <yxiva/core/json.h>

namespace yxiva::mailpusher {

namespace detail {

inline operation::result validate_account(const json_value& account)
{
    if (!account["uid"].is_string())
    {
        return "invalid uid (expected string)";
    }
    if (!account["environment"].is_string())
    {
        return "invalid environment (expected string)";
    }
    auto&& badge = account["badge"];
    if (!badge.is_object())
    {
        return "invalid badge (expected object)";
    }
    if (!badge["enabled"].is_bool())
    {
        return "invalid badge.enabled (expected bool)";
    }
    return {};
}

inline operation::result parse_exfid(const json_value& exfid, std::vector<uint64_t>& fids)
{
    if (!exfid.is_array())
    {
        return "invalid exfid (expected array)";
    }
    for (auto&& fid : exfid.array_items())
    {
        if (!fid.is_uint64())
        {
            return "invalid fid (expected uint64)";
        }
        fids.push_back(fid.to_uint64());
    }
    return {};
}

inline operation::result parse_account(const json_value& account_js, struct account& account)
{
    auto&& badge = account_js["badge"];
    account.uid = account_js["uid"].to_string();
    account.environment = account_js["environment"].to_string();
    account.badge_enabled = badge["enabled"].to_bool();
    if (badge.has_member("exfid"))
    {
        auto exfid_result = parse_exfid(badge["exfid"], account.exfid);
        if (!exfid_result)
        {
            return exfid_result;
        }
    }
    return {};
}

}

inline operation::result parse_accounts(const string& extra_string, std::vector<account>& accounts)
{
    json_value extra;
    if (!json_parse(extra, extra_string))
    {
        return "invalid json";
    }
    if (!extra["accounts"].is_array())
    {
        return "no accounts array found";
    }
    for (auto&& account_js : extra["accounts"].array_items())
    {
        auto account_valid = detail::validate_account(account_js);
        if (!account_valid)
        {
            return account_valid;
        }
        accounts.emplace_back();
        auto account_parsed = detail::parse_account(account_js, accounts.back());
        if (!account_parsed)
        {
            return account_parsed;
        }
    }
    return {};
}

}