#pragma once

#include "../mod_log/mod_log.h"
#include "avatar_fetcher.h"
#include "counters_fetcher.h"
#include "metadata_fetcher.h"
#include "notification_factory.h"
#include "notification_sender.h"
#include "searchapp_sender.h"
#include "settings.h"
#include "subscription_fetcher.h"
#include <mailpusher/task.h>
#include <mailpusher/types.h>
#include <ymod_ratecontroller/rate_controller.h>
#include <yplatform/coroutine.h>
#include <algorithm>

#include <boost/asio/yield.hpp>

namespace yxiva::mailpusher {
namespace {

template <typename Container, typename Predicate>
inline bool any_of(const Container& c, Predicate&& p)
{
    return std::any_of(c.begin(), c.end(), std::forward<Predicate>(p));
}

template <typename Container, typename Value>
inline bool contains(const Container& c, const Value& v)
{
    return std::find(c.begin(), c.end(), v) != c.end();
}

}

template <typename HttpClient>
struct processor
{
    using avatar_fetcher_t = avatar_fetcher<HttpClient>;
    using counters_fetcher_t = counters_fetcher<HttpClient>;
    using metadata_fetcher_t = metadata_fetcher<HttpClient>;
    using notification_sender_t = notification_sender<HttpClient, notification_factory>;
    using searchapp_sender_t = searchapp_sender<HttpClient>;
    using subscription_fetcher_t = subscription_fetcher<HttpClient>;
    using yield_context_t = yplatform::yield_context<processor<HttpClient>>;

    shared_ptr<task> task;
    const settings& settings;
    std::shared_ptr<std::map<string, HttpClient>> http_clients;
    std::shared_ptr<ymod_ratecontroller::rate_controller_module> rc;
    callback_t cb;
    std::shared_ptr<mod_log> log = std::make_shared<mod_log>(); // Defaults to test mock

    void operator()(yield_context_t yield_context, error_code ec = {})
    {
        task->profilers.pop();
        try
        {
            reenter(yield_context)
            {
                yield run_step<subscription_fetcher_t>(yield_context);
                // Fail to list subscriptions is treated as error.
                if (ec)
                {
                    log->failed(task, "list " + ec.message());
                    yield break;
                }
                else if (task->subscriptions.empty())
                {
                    task->processed = task->events.size();
                    for (auto& event : task->events)
                    {
                        log->dropped(task, event, "no subscriptions");
                    }
                    // yield break;
                }
                if (any_of(task->events, counters_fetcher_t::required))
                {
                    yield run_step<counters_fetcher_t>(yield_context);
                    // Fail to get counters is treated as warning by specification.
                    if (ec)
                    {
                        YLOG_CTX_GLOBAL(task, info) << "failed to get counters: " << ec.message();
                    }
                }
                for (; task->processed < task->events.size(); ++task->processed)
                {
                    // If current event requires metadata, fetch it for up to
                    // max_mids_in_filter_search_request mids (over one or multiple events).
                    while (metadata_fetcher_t::required(task->next_unprocessed_event()))
                    {
                        yield run_step<metadata_fetcher_t>(yield_context);
                        // Fail to get metadata is treated as error.
                        if (ec)
                        {
                            log->failed(task, "meta " + ec.message());
                            yield break;
                        }
                    }
                    // If current event requires avatar, fetch it for up to
                    // max_emails_in_profiles_request events.
                    if (avatar_fetcher_t::required(task->next_unprocessed_event(), settings))
                    {
                        yield run_step<avatar_fetcher_t>(yield_context);
                        // Fail to get avatar is treated as warning by specification.
                        if (ec)
                        {
                            YLOG_CTX_GLOBAL(task, info) << "failed to get avatar: " << ec.message();
                        }
                    }
                    // If required, compute badge for each device
                    if (task->next_unprocessed_event().action_type == action::NEW_MAIL)
                    {
                        compute_badge(task->next_unprocessed_event());
                    }
                    // If required, send notifications for exactly one event.
                    if (check_send_required())
                    {
                        yield run_step<notification_sender_t>(yield_context);
                        // Fail to send push is treated as error.
                        if (!ec)
                        {
                            log->sent(task, task->next_unprocessed_event());
                        }
                        else if (ec.value() == error::payload_too_large)
                        {
                            log->dropped(task, task->next_unprocessed_event(), ec.message());
                            // While notification sender failed if paylaod is too large,
                            // task was processed correctly.
                            ec = make_error(error::success);
                        }
                        else
                        {
                            log->failed(
                                task, task->next_unprocessed_event(), "send " + ec.message());
                            yield break;
                        }
                    }
                }
                // Finally send one summary notification to Yandex search app installations.
                yield run_step<searchapp_sender_t>(yield_context);
                ec = make_error(error::success);
            }
        }
        catch (const std::exception& ex)
        {
            log->failed(task, ex.what());
            ec = make_error(error::internal_error);
        }

        if (yield_context.is_complete())
        {
            cb(ec);
        }
    }

    template <typename Step>
    void run_step(yield_context_t yield_context)
    {
        task->profilers.push(Step::name());
        yplatform::spawn(Step{ task,
                               get_rate_controller((Step*){}),
                               get_http_client((Step*){}),
                               settings,
                               yield_context });
    }

    template <typename Step>
    ymod_ratecontroller::rate_controller_ptr get_rate_controller(Step*)
    {
        return rc->get_controller(Step::name());
    }

    auto get_rate_controller(subscription_fetcher_t*)
    {
        return get_rc_for_all_environments(subscription_fetcher_t::name(), settings.list);
    }

    auto get_rate_controller(counters_fetcher_t*)
    {
        return get_rc_for_all_environments(counters_fetcher_t::name(), settings.counters);
    }

    template <typename Step>
    HttpClient& get_http_client(Step*)
    {
        return http_clients->at(Step::name());
    }

    auto get_http_client(subscription_fetcher_t*)
    {
        return get_clients_for_all_environments(subscription_fetcher_t::name(), settings.list);
    }

    auto get_http_client(counters_fetcher_t*)
    {
        return get_clients_for_all_environments(counters_fetcher_t::name(), settings.counters);
    }

    template <typename FetcherSettings>
    auto get_rc_for_all_environments(
        const string& fetcher_name,
        const FetcherSettings& fetcher_settings)
    {
        rate_controllers<ymod_ratecontroller::rate_controller_ptr> controllers;
        for (auto& environment_settings : fetcher_settings)
        {
            auto& environment = environment_settings.first;
            controllers.emplace(environment, rc->get_controller(fetcher_name + "." + environment));
        }
        return controllers;
    }

    template <typename FetcherSettings>
    auto get_clients_for_all_environments(
        const string& fetcher_name,
        const FetcherSettings& fetcher_settings)
    {
        yxiva::mailpusher::http_clients<HttpClient&> clients;
        for (auto& environment_settings : fetcher_settings)
        {
            auto& environment = environment_settings.first;
            clients.emplace(environment, http_clients->at(fetcher_name + "." + environment));
        }
        return clients;
    }

    // Drop reason is checked and logged in a separate function because of
    // boost::asio::coroutine limitations on variabliable initialization.
    bool check_send_required()
    {
        auto send_required =
            notification_sender_t::required(task->next_unprocessed_event(), settings);
        if (!send_required)
        {
            log->dropped(task, task->next_unprocessed_event(), send_required.error_reason);
        }
        return send_required;
    }

    void compute_badge(event& event)
    {
        for (auto& [subscription_id, device] : task->devices)
        {
            // Compute only for multi-account APNS devices.
            if (device.platform != platform::APNS || device.accounts.size() < 2)
            {
                continue;
            }
            if (auto badge = compute_badge(device))
            {
                event.args["mailpusher_badge"][subscription_id] = *badge;
            }
        }
    }

    std::optional<uint64_t> compute_badge(const device& device)
    {
        uint64_t badge = 0;
        for (auto& account : device.accounts)
        {
            // If any necessary data was not obtained,
            // do not send any badge. Otherwise users may
            // observe badge randomly changing values.
            if (indeterminate(account.has_subscriptions) ||
                account.counters_fetched != fetch_status::FETCHED)
            {
                return {};
            }
            if (!account.has_subscriptions || !account.badge_enabled)
            {
                continue;
            }
            for (auto& counters : account.fids_counters)
            {
                if (!contains(account.exfid, counters.fid))
                {
                    badge += counters.count_new;
                }
            }
        }
        return badge;
    }
};

}

#include <boost/asio/unyield.hpp>